#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/libharkio3/xml.py

"""

import typing
import xml.etree.ElementTree as ET
from xml.etree.ElementTree import Element

from .config import Config as _Config, ConfigParser
from .defs import UseTagType
from .error import LibHarkIOError
from .neighbors import Neighbors, NeighborsParser
from .positions import Positions, PositionsParser
import pydantic

class HarkXML(pydantic.BaseModel):
    class Config:
        arbitrary_types_allowed = True
        
    positions: Positions | None = None
    config: _Config | None = None
    neighbors: Neighbors | None = None
    

class HarkXMLParser:
    _HARKIO_VER = '1.3'

    @classmethod
    def as_file(cls, xml: HarkXML, file: str | typing.IO) -> None:
        if isinstance(file, str):
            with open(file, 'wb') as fp:
                return HarkXMLParser.as_file(xml, fp)

        tree = ET.ElementTree(HarkXMLParser.as_element(xml))
        ET.indent(tree, space="    ", level=0)
        tree.write(file, encoding='utf-8', xml_declaration=True)

    @classmethod
    def as_element(cls, xml: HarkXML) -> Element:
        hark_xml = ET.Element('hark_xml', attrib={
                              'version': HarkXMLParser._HARKIO_VER})
        if xml.positions:
            hark_xml.append(PositionsParser.as_element(xml.positions))
        if xml.config:
            hark_xml.append(ConfigParser.as_element(xml.config))
        if xml.neighbors:
            hark_xml.append(NeighborsParser.as_element(xml.neighbors))

        if xml.positions:
            PositionsParser.append_channels_element(xml.positions, hark_xml)
        return hark_xml

    @classmethod
    def from_element(cls, element: Element) -> HarkXML:
        new_positions:Positions = None
        new_config:Config = None
        new_neighbors:Neighbors = None
        for child in element:
            if child.tag == 'positions':
                if new_positions is None:
                    new_positions = PositionsParser.from_element(child)
                else:
                    raise LibHarkIOError(f'Duplicate <{child.tag}> in XML')

            elif child.tag == 'config':
                if new_config is None:
                    new_config = ConfigParser.from_element(child)
                else:
                    raise LibHarkIOError(f'Duplicate <{child.tag}> in XML')

            elif child.tag == 'neighbors':
                if new_neighbors is None:
                    new_neighbors = NeighborsParser.from_element(child)
                else:
                    raise LibHarkIOError(f'Duplicate <{child.tag}> in XML')

            elif child.tag == 'channels':
                channel_ids = [int(s) for s in child.attrib.get('use', '').split(';') if s]
                new_positions[0].channels_use = UseTagType.USE_TAG
                new_positions[0].channel_ids = channel_ids

            elif child.tag == 'channel':
                source_id = int(child.attrib.get('id'))
                channel_ids = [int(s) for s in child.attrib.get('use', '').split(';') if s]

                position = new_positions.query_position_by_id(source_id)
                position.channels_use = UseTagType.ID_TAG
                position.channel_ids = channel_ids

        return HarkXML(positions=new_positions, config=new_config, neighbors=new_neighbors)

    @classmethod
    def from_file(cls, file: str | typing.IO) -> HarkXML:
        if isinstance(file, str):
            with open(file, 'r') as fp:
                return HarkXMLParser.from_file(fp)

        root = ET.parse(file).getroot()
        if root.tag != 'hark_xml':
            raise LibHarkIOError('No <hark_xml> in XML')
        else:
            return HarkXMLParser.from_element(root)

# end of file
