#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/libharkio3/config.py

"""

import xml.etree.ElementTree as ET
from xml.etree.ElementTree import Element

from .error import LibHarkIOError

import pydantic

class Config(pydantic.BaseModel):
    class Config:
        arbitrary_types_allowed = True

    comment: str = None
    synchronous_average:pydantic.conint(ge=0) = 0
    tsp_path:str = None
    tsp_offset:int = 0
    peak_search:tuple[int,int] = (0, 0)
    n_fft:pydantic.conint(ge=0) = 512
    sampling_rate:pydantic.conint(ge=0) = 16000
    tsp_length:int = 0
    signal_max:int = 0

    def __repr__(self) -> str:
        return \
            """Config: 
    comment: {comment}-
    syncavg: {synchronous_average}
    offset : {tsp_offset}
    path   : {tsp_path}-
    from   : {peak_search[0]}
    to     : {peak_search[1]}
    nfft   : {n_fft}
    sample : {sampling_rate}
    TSPLength : {tsp_length}
    signalMax : {signal_max}""".format(**vars(self))


class ConfigParser:
    _config_attributes = dict(
        comment={'att_name': 'comment', 'att_type': str},
        SynchronousAverage={
            'att_name': 'synchronous_average', 'att_type': int},
        TSPpath={'att_name': 'tsp_path', 'att_type': str},
        TSPOffset={'att_name': 'tsp_offset', 'att_type': int},
        nfft={'att_name': 'n_fft', 'att_type': int},
        samplingRate={'att_name': 'sampling_rate', 'att_type': int},
        signalMax={'att_name': 'signal_max', 'att_type': int},
        TSPLength={'att_name': 'tsp_length', 'att_type': int},
    )

    @classmethod
    def as_element(cls, config: Config) -> Element:
        element = ET.Element('config')

        for key, value in ConfigParser._config_attributes.items():
            child = ET.Element(key)
            child.text = str(
                getattr(config, value['att_name']))
            element.append(child)

        element.append(ET.Element('PeakSearch', {
            'from': str(config.peak_search[0]),
            'to':   str(config.peak_search[1]),
        }))

        return element

    @classmethod
    def from_element(cls, element: Element) -> Config:
        if element.tag != 'config':
            raise LibHarkIOError(f'Unexpected element <{element.tag}>.')

        new_config = Config()
        for child in element:
            if attr := ConfigParser._config_attributes.get(child.tag):
                setattr(new_config, attr['att_name'],
                        attr['att_type'](child.text))

            elif child.tag == 'peaksearch':
                ps_from = child.attrib.get('from', 0)
                ps_to = child.attrib.get('to', 0)
                new_config.peak_search = (ps_from, ps_to)

        return new_config

# end of file
