import argparse
import sys
import types
import typing

import pandas

from .. import utils
from ..libharkio3 import TransferFunctionParser
from ..libharkio3.transferfunction import _DTYPE_TO_DATATYPES, _HARKIO_TAG
from .defs import LogLevel
from .utils import StoreLogLevelAction

_default_values = types.SimpleNamespace(
    show_header=False,
    show_as_exp=False,
    log_level=LogLevel.Error,
)


def parse_args(args):
    parser = argparse.ArgumentParser(
        description='Convert csv file from mat file.')

    parser.add_argument('--header', '-H', required=False,
                        dest='show_header', action='store_true',
                        default=_default_values.show_header,
                        help='Show header')
    parser.add_argument('--show-exponent', '-e', required=False,
                        dest='show_as_exp', action='store_true',
                        default=_default_values.show_as_exp,
                        help='Show float as exponent notation')
    parser.add_argument('--log-level', '--llevel', metavar='{E|W|I|D}', required=False,
                        type=LogLevel, action=StoreLogLevelAction, dest='log_level',
                        default=_default_values.log_level,
                        help='Log information level. (default=%(default)s)', )
    parser.add_argument(metavar='INFILE',
                        type=str, dest='input_file',
                        help='Input matrix file')
    parser.add_argument(metavar='OUTFILE',
                        type=str, dest='output_file',
                        help='Output csv file')
    return parser.parse_args(args)


def main(args=sys.argv[1:]):
    args = parse_args(args)

    logger = utils.initialize_logger(args)

    matrix_to_csv(**vars(args), logger=logger)


def matrix_to_csv(
    show_header: bool,
    show_as_exp: bool,
    input_file: str,
    output_file: str,
    logger: typing.Any,
    **kwargs,
) -> None:

    # load matrix
    with open(input_file, 'rb') as fp:
        matrix = TransferFunctionParser.read_matrix(fp)

    # save csv
    with open(output_file, 'w') as fp:
        # write header
        if show_header:
            print(','.join(str(v) for v in [_HARKIO_TAG, _DTYPE_TO_DATATYPES[matrix.dtype].value, matrix.ndim] + list(matrix.shape)), file=fp)

        # write matrix
        df = pandas.DataFrame(matrix.view(float).tolist())
        df.to_csv(fp, index=False, header=False, float_format='%e' if show_as_exp else '%f')

# end of file
