#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/app/genposxml.py

"""

import argparse
import itertools
import logging
import sys
import types
import typing

import numpy

from .. import utils
from ..libharkio3 import HarkXML, HarkXMLParser, Position, Positions
from ..libharkio3.defs import CoordinateSystem, UseTagType
from .defs import LogLevel, PositionType
from .error import HarkToolError
from .utils import StoreLogLevelAction, StorePositionsTypeAction

_default_values = types.SimpleNamespace(
    position_type = PositionType.SOUNDSOURCE,
    channel_id_origin = 0,
    log_level = LogLevel.Error,
)

def parse_args(args):
    parser = argparse.ArgumentParser(
        description='create a position tag of xml file.')
    subparsers = parser.add_subparsers(
        dest='modelname', help='Coordinate model')

    parser_circle = subparsers.add_parser('circle', description='create a position tag of xml file',
                                          epilog='coordinate model is circle.')
    parser_circle.add_argument('--elevation', type=float, metavar='NUM',
                               required=True, dest='elevation_value', help='Value of elevation')
    parser_circle.add_argument('--min-azimuth', type=float, metavar='NUM',
                               required=True, dest='azimuth_min', help='Minimum of azimuth')
    parser_circle.add_argument('--max-azimuth', type=float, metavar='NUM',
                               required=True, dest='azimuth_max', help='Maximum of azimuth')
    parser_circle.add_argument('--interval-azimuth', type=float, metavar='NUM',
                               required=True, dest='azimuth_interval', help='Interval of azimuth')
    parser_circle.add_argument('--min-radius', type=float, metavar='NUM',
                               required=True, dest='radius_min', help='Minimum of radius')
    parser_circle.add_argument('--max-radius', type=float, metavar='NUM',
                               required=True, dest='radius_max', help='Maximum of radius')
    parser_circle.add_argument('--interval-radius', type=float, metavar='NUM',
                               required=True, dest='radius_interval', help='Interval of radius')

    parser_grid = subparsers.add_parser('grid', description='create a position tag of xml file',
                                        epilog='coordinate model is grid.')
    parser_grid.add_argument('--min-x-axis', type=float, metavar='NUM',
                             required=True, dest='x_axis_min', help='Minimum of x-axis')
    parser_grid.add_argument('--max-x-axis', type=float, metavar='NUM',
                             required=True, dest='x_axis_max', help='Maximum of x-axis')
    parser_grid.add_argument('--interval-x-axis', type=float, metavar='NUM',
                             required=True, dest='x_axis_interval', help='Interval of x-axis')
    parser_grid.add_argument('--min-y-axis', type=float, metavar='NUM',
                             required=True, dest='y_axis_min', help='Minimum of y-axis')
    parser_grid.add_argument('--max-y-axis', type=float, metavar='NUM',
                             required=True, dest='y_axis_max', help='Maximum of y-axis')
    parser_grid.add_argument('--interval-y-axis', type=float, metavar='NUM',
                             required=True, dest='y_axis_interval', help='Interval of y-axis')
    parser_grid.add_argument('--min-z-axis', type=float, metavar='NUM',
                             required=True, dest='z_axis_min', help='Minimum of z-axis')
    parser_grid.add_argument('--max-z-axis', type=float, metavar='NUM',
                             required=True, dest='z_axis_max', help='Maximum of z-axis')
    parser_grid.add_argument('--interval-z-axis', type=float, metavar='NUM',
                             required=True, dest='z_axis_interval', help='Interval of z-axis')

    parser_cylinder = subparsers.add_parser('cylinder', description='create a position tag of xml file',
                                            epilog='coordinate model is cylinder.')
    parser_cylinder.add_argument('--min-azimuth', type=float, metavar='NUM',
                                 required=True, dest='azimuth_min', help='Minimum of azimuth')
    parser_cylinder.add_argument('--max-azimuth', type=float, metavar='NUM',
                                 required=True, dest='azimuth_max', help='Maximum of azimuth')
    parser_cylinder.add_argument('--interval-azimuth', type=float, metavar='NUM',
                                 required=True, dest='azimuth_interval', help='Interval of azimuth')
    parser_cylinder.add_argument('--min-radius', type=float, metavar='NUM',
                                 required=True, dest='radius_min', help='Minimum of radius')
    parser_cylinder.add_argument('--max-radius', type=float, metavar='NUM',
                                 required=True, dest='radius_max', help='Maximum of radius')
    parser_cylinder.add_argument('--interval-radius', type=float, metavar='NUM',
                                 required=True, dest='radius_interval', help='Interval of radius')
    parser_cylinder.add_argument('--min-height', type=float, metavar='NUM',
                                 required=True, dest='height_min', help='Minimum of height')
    parser_cylinder.add_argument('--max-height', type=float, metavar='NUM',
                                 required=True, dest='height_max', help='Maximum of height')
    parser_cylinder.add_argument('--interval-height', type=float, metavar='NUM',
                                 required=True, dest='height_interval', help='Interval of height')

    parser_ball = subparsers.add_parser('ball', description='create a position tag of xml file',
                                        epilog='coordinate model is ball.')
    parser_ball.add_argument('--min-elevation', type=float, metavar='NUM',
                             required=True, dest='elevation_min', help='Minimum of elevation')
    parser_ball.add_argument('--max-elevation', type=float, metavar='NUM',
                             required=True, dest='elevation_max', help='Maximum of elevation')
    parser_ball.add_argument('--interval-elevation', type=float, metavar='NUM',
                             required=True, dest='elevation_interval', help='Interval of elevation')
    parser_ball.add_argument('--min-azimuth', type=float, metavar='NUM',
                             required=True, dest='azimuth_min', help='Minimum of azimuth')
    parser_ball.add_argument('--max-azimuth', type=float, metavar='NUM',
                             required=True, dest='azimuth_max', help='Maximum of azimuth')
    parser_ball.add_argument('--interval-azimuth', type=float, metavar='NUM',
                             required=True, dest='azimuth_interval', help='Interval of azimuth')
    parser_ball.add_argument('--min-radius', type=float, metavar='NUM',
                             required=True, dest='radius_min', help='Minimum of radius')
    parser_ball.add_argument('--max-radius', type=float, metavar='NUM',
                             required=True, dest='radius_max', help='Maximum of radius')
    parser_ball.add_argument('--interval-radius', type=float, metavar='NUM',
                             required=True, dest='radius_interval', help='Interval of radius')

    parser.add_argument('--type', '-t', type=PositionType, action=StorePositionsTypeAction, metavar='NAME', default=_default_values.position_type, required=False, dest='position_type',
                        help='Type of positions --type=tsp|impulse|noise|microphone|soundsource (default=%(default)s)')
    parser.add_argument('--output', '-o', type=str, metavar='PATH',
                        required=True, help='Output filename.', dest='output_file')
    parser.add_argument('--id', '-i', type=int, metavar='NUM', default=_default_values.channel_id_origin,
                        help='Position id of start. (default=%(default)s)', dest='channel_id_origin')
    parser.add_argument('--path', '-p', type=str,
                        metavar='PATH', required=True, dest='path_format')
    parser.add_argument('--channels', '-c', type=int, nargs='+', metavar='NUM',
                        dest='channels', help='Select microphone channel(s) to calculate. (zero-based)')
    parser.add_argument('--logLevel', type=LogLevel, action=StoreLogLevelAction, metavar='{E|W|I|D}', default=_default_values.log_level,
                        required=False, help='Log information level. (default=%(default)s)', dest='log_level')

    return parser.parse_args(args)


def main(args=sys.argv[1:]):
    args = parse_args(args)

    logger = utils.initialize_logger(args)
    logger.info('Args: %s', args)
    
    # verify args
    # if args.posestype not in ('tsp', 'impulse', 'noise', 'microphone', 'soundsource'):
    #     raise HarkToolError('invalid value for --type/-t option.')
    if args.channel_id_origin < 0:
        raise HarkToolError('Unexpected value for --id/-i.')
    if args.channels is not None and len([c for c in args.channels if c < 0]) > 0: # or c > _MAX_CH]) > 0:
        raise HarkToolError('Unexpeted channel index for --channels/-c.')
    if hasattr(args, 'radius_min') and args.radius_min < 0:
        raise HarkToolError('Unexpected value for --radius-min.')
    if hasattr(args, 'radius_max') and args.radius_max < 0:
        raise HarkToolError('Unexpected value for --radius-max.')
    if (value := getattr(args, 'elevation_value', None)) is not None:
        setattr(args, 'elevation_min', value)
        setattr(args, 'elevation_max', value)
        setattr(args, 'elevation_interval', 0)

    for name in ('radius', 'elevation', 'height', 'azimuth', 'x_axis', 'y_axis', 'z_axis'):
        vmin = getattr(args, f'{name}_min', None)
        vmax = getattr(args, f'{name}_max', None)
        if vmin is not None and vmax is not None and vmin > vmax:
            raise HarkToolError(f'--min-{name} should not be greater than --max-{name}')

    if args.position_type == PositionType.MICROPHONE:
        if hasattr(args, 'channels') and args.channels is not None:
            raise HarkToolError('Microphone should not have --channels')
        if args.channel_id_origin != 0:
            raise HarkToolError('Microphone should have --id == 0')

    # verify args: end

    # make positions
    poses = Positions(type=args.position_type)

    _make_positions_from_model(args, poses, args.channels, logger)

    # if args.position_type == PositionType.MICROPHONE and len(poses) > _MAX_CH:
    #     raise HarkToolError(
    #         'Microphone position too large. max {}'.format(_MAX_CH))

    xml = HarkXML(positions=poses)
    HarkXMLParser.as_file(xml, args.output_file)

    logger.debug('XML file created')


def _make_positions_from_model(args: typing.Any, poses: Positions, channels: list[int], logger: logging.Logger) -> None:
    # print('args.id: {}'.format(args.id))
    channel_id: int = args.channel_id_origin
    if channels is None:
        channels = []

    values = {}
    for name in ('radius', 'elevation', 'height', 'azimuth', 'x_axis', 'y_axis', 'z_axis'):
        vmin = getattr(args, '{}_min'.format(name), None)
        vmax = getattr(args, '{}_max'.format(name), None)
        vint = getattr(args, '{}_interval'.format(name), None)
        if vmin is None or vmax is None or vint is None:
            continue

        if vint > 0:
            vals = numpy.arange(vmin, vmax + vint, vint)
            while vals[-1] > vmax:
                vals = vals[:-1]
        elif vint < 0:
            vals = numpy.arange(vmax, vmin - vint, -vint)
            while vals[-1] < vmin:
                vals = vals[:-1]
            vals = vals[::-1]
        else:
            vals = numpy.array([vmin])

        # print('min={}, max={}, vint={}, vals={}'.format(vmin, vmax, vint, vals))

        values[name] = vals

    is_channel_set: bool = (len(channels) == 0)
    if args.modelname in ('circle', 'ball'):
        for radius, elevation, azimuth in itertools.product(values['radius'], values['elevation'], values['azimuth']):
            logger.debug('[{}] azimuth[{:.4f}] elevation[{:.4f}] radius[{:.4f}]'.format(
                channel_id, azimuth, elevation, radius))

            path = _make_file_path(
                args.path_format, radius=radius, elevation=elevation, azimuth=azimuth)
            # print('id={}'.format(id))
            pos = Position(position_id=channel_id, coordinate_system=CoordinateSystem.Cartesian, coordinate=_polar_to_cartesian(
                azimuth, elevation, radius), path=path)

            if not is_channel_set:
                pos.channels = channels
                pos.channels_use = UseTagType.USE_TAG
                is_channel_set = True

            poses += pos
            channel_id += 1

    elif args.modelname in ('cylinder', ):
        for radius, height, azimuth in itertools.product(values['radius'], values['height'], values['azimuth']):
            logger.debug('[{}] azimuth[{}] height[{}] radius[{}]'.format(
                id, azimuth, height, radius))

            path = _make_file_path(
                args.path_format, radius=radius, height=height, azimuth=azimuth)
            pos = Position(channel_id, CoordinateSystem.Cartesian, _cylindrical_to_cartesian(
                azimuth, radius, height), path)

            if not is_channel_set:
                pos.channels = channels
                pos.channels_use = UseTagType.USE_TAG
                is_channel_set = True

            poses += pos
            channel_id += 1

    elif args.modelname in ('grid', ):
        for z, y, x in itertools.product(values['x'], values['y'], values['z']):
            logger.debug('[{}] = ({}, {}, {})'.format(id, x, y, z))

            path = _make_file_path(args.path_format, x_axis=x, y_axis=y, z_axis=z)
            pos = Position(
                channel_id, CoordinateSystem.Cartesian, (x, y, z), path)

            if not is_channel_set:
                pos.channels = channels
                pos.channels_use = UseTagType.USE_TAG
                is_channel_set = True

            poses += pos
            channel_id += 1

    else:
        raise HarkToolError(
            'Unexpected coordinate model: {}'.format(args.modelname))

def _polar_to_cartesian(azimuth: float, elevation: float, radius: float) -> tuple[float, float, float]:
    azimuth_rad = numpy.deg2rad(azimuth)
    elevation_rad = numpy.deg2rad(elevation)

    c = radius * numpy.cos(elevation_rad)
    x = c * numpy.cos(azimuth_rad)
    y = c * numpy.sin(azimuth_rad)
    z = radius * numpy.sin(elevation_rad)

    return (x, y, z)


def _cylindrical_to_cartesian(azimuth: float, radius: float, height: float) -> tuple[float, float, float]:
    azimuth_rad = numpy.deg2rad(azimuth)

    x = radius * numpy.cos(azimuth_rad)
    y = radius * numpy.sin(azimuth_rad)
    z = height

    return (x, y, z)


def _make_file_path(path: str, **kwargs) -> str:
    args = dict((k.upper(), v) for (k, v) in kwargs.items())
    return path.format(**args)

# end of file
