#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/app/genconfig.py

"""

import argparse
import os
import sys

from .. import utils
from ..libharkio3 import Config, HarkXML, HarkXMLParser
from .utils import StoreLogLevelAction
from .defs import LogLevel
import types

_default_values = types.SimpleNamespace(
    comment = 'comment',
    synchronous_average = 16,
    tsp_offset = 0,
    peak_search = (0, 0),
    n_fft = 512,
    sampling_rate = 16000,
    signal_max = 32768,
    tsp_length = 16384,
    tsp_path = os.path.join('.', r'16384.little_endian.wav'),
    log_level = LogLevel.Error,
)


def parse_args(args):
    parser = argparse.ArgumentParser(
        description='The command creates a xml file that contians a config tag.')

    parser.add_argument('--comment', '-c', type=str, metavar='STR', dest='comment',
                        default=_default_values.comment, help='Comment. (default=%(default)s)')
    parser.add_argument('--sync', '-s', type=int, metavar='NUM', dest='synchronous_average',
                        default=_default_values.synchronous_average, help='Number of TSPs in the recorded wav. (default=%(default)s)')
    parser.add_argument('--offset', '-o', type=float, metavar='NUM', dest='tsp_offset',
                        default=_default_values.tsp_offset, help='Offset of the TSP. (default=%(default)s)')
    parser.add_argument('--peak', '-p', type=int, nargs='*', metavar='NUM', dest='peak_search',
                        default=_default_values.peak_search, help='Range for peak search. (default=%(default)s)')
    parser.add_argument('--nfft', '-n', type=int, metavar='NUM', dest='n_fft',
                        default=_default_values.n_fft, help='NFFT of convolution. (default=%(default)s)')
    parser.add_argument('--rate', '-r', type=int, metavar='NUM', dest='sampling_rate',
                        default=_default_values.sampling_rate, help='SamplingRate. (default=%(default)s)')
    parser.add_argument('--max', '-m', type=int, metavar='NUM', dest='signal_max',
                        default=_default_values.signal_max, help='SignalMax. (default=%(default)s)')
    parser.add_argument('--length', '-l', type=int, metavar='NUM',
                        dest='tsp_length', default=_default_values.tsp_length, help='TSPlength. (default=%(default)s)')
    parser.add_argument('--tsp-path', '-t', type=str, metavar='PATH', dest='tsp_path',
                        default=_default_values.tsp_path, help='Path to the original TSP. (default=%(default)s)')
    parser.add_argument('--logLevel', type=LogLevel, action=StoreLogLevelAction, metavar='{E|W|I|D}', default=_default_values.log_level,
                        required=False, help='Log information level. (default=%(default)s)', dest='log_level')
    parser.add_argument(type=str, metavar='OUTFILE',
                        help='Output file name.', dest='output_file')

    return parser.parse_args(args)


def main(args=sys.argv[1:]):
    args = parse_args(args)

    logger = utils.initialize_logger(args)
    logger.info('Args: %s', args)
    
    # verify args
    # verify args: end

    # make positions
    config = Config()
    config.comment = args.comment
    config.synchronous_average = args.synchronous_average
    config.tsp_offset = args.tsp_offset
    config.peak_search = args.peak_search
    config.n_fft = args.n_fft
    config.tsp_path = args.tsp_path
    config.sampling_rate = args.sampling_rate
    config.signal_max = args.signal_max
    config.tsp_length = args.tsp_length

    xml = HarkXML(config=config)
    HarkXMLParser.as_file(xml, args.output_file)

    logger.debug('XML file created')


# end of file
