import argparse
import sys
import types
import typing

from .. import utils
from .defs import LogLevel
from .error import HarkToolError
from .utils import StoreLogLevelAction
from .workset import SoundData

_default_values = types.SimpleNamespace(
    list_file=None,
    sampling_rate=16000,
    log_level=LogLevel.Error,
)


def parse_args(args):
    parser = argparse.ArgumentParser(
        description='Convert the multiple raw data (32bit float wave) to a multi channel wave file.')

    parser.add_argument('--list', '-l', required=False,
                        type=str, dest='list_file',
                        default=_default_values.list_file,
                        help='Input_raw_listfilename (ex. list.txt)')
    parser.add_argument('--sampling-rate', '-r', required=False,
                        type=int, dest='sampling_rate',
                        default=_default_values.sampling_rate,
                        help='Sampling rate (default=%(default)s)')
    parser.add_argument('--log-level', '--llevel', metavar='{E|W|I|D}', required=False,
                        type=LogLevel, action=StoreLogLevelAction, dest='log_level',
                        default=_default_values.log_level,
                        help='Log information level. (default=%(default)s)', )
    parser.add_argument(metavar='OUTFILE',
                        type=str, dest='output_file',
                        help='Output wav file')
    parser.add_argument(metavar='INFILE',
                        type=str, dest='input_files', nargs='*',
                        help='Input raw file')
    return parser.parse_args(args)


def main(args=sys.argv[1:]):
    args = parse_args(args)
    logger = utils.initialize_logger(args)

    if args.list_file and not args.input_files:
        with open(args.list_file, 'r') as fp:
            input_files = fp.readlines()
    elif args.input_files and not args.list_file:
        input_files = args.input_files
    elif args.list_file and args.input_files:
        raise HarkToolError()
    else:
        raise HarkToolError()

    floats_to_multi_channel_wav(input_files=input_files,
                                output_file=args.output_file,
                                sampling_rate=args.sampling_rate,
                                logger=logger)


def floats_to_multi_channel_wav(
    input_files: list[str],
    output_file: str,
    sampling_rate: int,
    logger: typing.Any,
    **kwargs,
) -> None:

    # load raw files
    wav = SoundData(sampling_rate=sampling_rate)
    for input_file in input_files:
        raw = SoundData.from_raw(input_file, sampling_rate=sampling_rate)
        wav.append(raw.waveform)

    # save wav
    SoundData.as_wav(output_file, wav)

# end of file
