import types
import typing

from .. import utils
from ..libharkio3 import (Config, HarkXML, HarkXMLParser, Neighbors, Positions,
                          TransferFunction, TransferFunctionParser)
from .calctf import IRFromTSP, IRVectorExtractor, TFExtractor
from .defs import LogLevel, TFType
from .error import HarkToolError
from .utils import (StoreLogLevelAction, StoreTFTypeAction,
                    select_output_channels)
from .workset import SoundData

_default_values = types.SimpleNamespace(
    map_path=None,
    output_type=TFType.LOC_SEP,
    direct_length=32,
    reverb_length=256,
    normalize_source=False,
    normalize_microphone=False,
    normalize_frequency=False,
    reset_microphone=False,
    log_level=LogLevel.Error,
)


def setup_parser(parser):
    parser.add_argument('--tsp-list', '--tlist', '-i', metavar='PATH', required=True,
                        type=str, dest='tsp_list',
                        help='TSP list file  (eg. tsp.xml)')
    parser.add_argument('--map-path', '--mapp', metavar='PATH1:PATH2', required=False,
                        type=str, dest='map_path', default=_default_values.map_path,
                        help='Replace PATH1 in path attribute with PATH2')
    parser.add_argument('--microphone-list', '--mlist', '-m', metavar='PATH', required=True,
                        type=str, dest='microphone_list',
                        help='Microphone list file  (eg. microphone.xml)')
    parser.add_argument('--output-type', '--otype', '-t', metavar='|'.join(e.name for e in TFType), required=False, default=_default_values.output_type,
                        type=TFType, action=StoreTFTypeAction, dest='output_type',
                        help='Transfer function type  LOC=Localization,SEP=Separation,LOC_SEP=LOC&SEP (default=%(default)s)')
    parser.add_argument('--output-file', '--ofile', '-o', metavar='PATH', required=True,
                        type=str, dest='output_file',
                        help='Output transfer function zip file path  (eg. /path/to/transfer.zip)')
    parser.add_argument('--direct-length', '--dlen', '-d', metavar='NUM', required=False, default=_default_values.direct_length,
                        type=int, dest='direct_length',
                        help='Direct sound length')
    parser.add_argument('--reverb-length', '--rlen', '-r', metavar='NUM', required=False, default=_default_values.reverb_length,
                        type=int, dest='reverb_length',
                        help='Reverb sound length')
    parser.add_argument('--normalize-source', '--snorm', '-S', metavar='True|False', required=False,
                        type=bool, dest='normalize_source',
                        default=_default_values.normalize_source, help='Normalize by source')
    parser.add_argument('--normalize-microphone', '--mnorm', '-M', metavar='True|False', required=False, default=_default_values.normalize_microphone,
                        type=bool,
                        help='Normalize by microphone', dest='normalize_microphone')
    parser.add_argument('--normalize-frequency', '--fnorm', '-F', metavar='True|False', required=False, default=_default_values.normalize_frequency,
                        type=bool, dest='normalize_frequency',
                        help='Normalize by frequency')
    parser.add_argument('--reset-microphone', '--mreset', '-R', default=_default_values.reset_microphone,
                        action='store_true', dest='reset_microphone',
                        help='Remove unused channels from microphones.xml and reset their ids. (for HARKTOOL4 compatible)')
    parser.add_argument('--log-level', '--llevel', metavar='{E|W|I|D}', required=False, default=_default_values.log_level,
                        type=LogLevel, action=StoreLogLevelAction, dest='log_level',
                        help='Log information level. (default=%(default)s)')
    parser.set_defaults(handler=main)


def main(args):
    logger = utils.initialize_logger(args)
    logger.info('harktool_calctfrecExec start')

    calculate_tf_from_recorded_tsp(**vars(args), logger=logger)


def calculate_tf_from_recorded_tsp(
    tsp_list: str, map_path: str, microphone_list: str, output_type: str, output_file: str,
    direct_length: int, reverb_length: int, normalize_source: bool, normalize_microphone: bool, normalize_frequency: bool, reset_microphone: bool,
    logger: typing.Any,
    **kwargs,
) -> None:
    
    # map path
    if map_path:
        map_path_from, map_path_to = map_path.split(':')
        path_mapper = lambda path: path.replace(map_path_from, map_path_to)
    else:
        path_mapper = lambda path: path

    # read XML files
    try:
        srcFileXML: HarkXML = HarkXMLParser.from_file(tsp_list)
    except BaseException as ex:
        raise HarkToolError(
            'Failed to read impulse list file: {}'.format(tsp_list)) from ex

    try:
        micFileXML: HarkXML = HarkXMLParser.from_file(microphone_list)
    except BaseException as ex:
        raise HarkToolError(
            'Failed to read microphone list file: {}'.format(microphone_list)) from ex

    # retrieve data
    source_neighbors: Neighbors = srcFileXML.neighbors  # may be None

    source_config: Config = srcFileXML.config
    if source_config is None:
        raise HarkToolError('Config not found in source xml')

    source_positions: Positions = srcFileXML.positions
    if source_positions is None:
        raise HarkToolError('Positions not found in source xml')

    microphone_positions: Positions = micFileXML.positions
    if microphone_positions is None:
        raise HarkToolError('Positions not found in microphone xml')

    # sort source positions
    source_positions.sort_positions_by_id()

    # extract output channels
    output_channel_indices = select_output_channels(source_positions, microphone_positions)

    irFromTSP = IRFromTSP(sampling_rate=source_config.sampling_rate, samples_by_frame=source_config.tsp_length,
                          sync_add_frequency=source_config.synchronous_average, sync_add_offset=source_config.tsp_offset, signal_maximum=source_config.signal_max)
    extractIRVect = IRVectorExtractor(
        output_type, direct_length, reverb_length, (-1, -1), source_config.n_fft)
    extractTFs = TFExtractor(output_type=output_type,
                             normalize_source=normalize_source, normalize_microphone=normalize_microphone, normalize_frequency=normalize_frequency,
                             reset_microphone=reset_microphone)

    source_ir_vectors = []
    for source_position in source_positions.positions:
        impulse_response = irFromTSP(tsp_original_signal=SoundData.from_wav(path_mapper(source_position.path)), tsp_original_channel=0, input_signal=SoundData.from_wav(
            file=path_mapper(source_position.path)), output_channels=output_channel_indices, search_offset=(-1, -1))
        source_ir_vectors.append(extractIRVect(impulse_response))

    loc_tfs, sep_tfs = extractTFs(source_ir_vectors)

    tf = TransferFunction(positions=source_positions, microphones=microphone_positions,
                          config=source_config, neighbors=source_neighbors, loc_tfs=loc_tfs, sep_tfs=sep_tfs)
    TransferFunctionParser.as_zipfile(tf, output_file)

# end of file
