import types
import typing

from .. import utils
from ..libharkio3 import (Config, HarkXML, HarkXMLParser, Neighbors, Positions,
                          TransferFunction, TransferFunctionParser)
from .calctf import IRFromMouthTSP, IRVectorExtractor, TFExtractor
from .defs import LogLevel, TFType
from .error import HarkToolError
from .utils import (StoreLogLevelAction, StoreTFTypeAction,
                    select_output_channels)
from .workset import SoundData, WorkingDirectory

_default_values = types.SimpleNamespace(
    map_path=None,
    output_type=TFType.LOC_SEP,
    direct_length=32,
    reverb_length=256,
    normalize_source=False,
    normalize_microphone=False,
    normalize_frequency=False,
    reset_microphone=False,
    log_level=LogLevel.Error,
    tsp_end=-1,
    voice_channel=-1,
    shift_length=1000,
    head_margin=0,
)


def setup_parser(parser):
    parser.add_argument('--tsp-list', '--tlist', '-i', metavar='PATH', required=True,
                        type=str, dest='tsp_list',
                        help='TSP list file  (eg. tsp.xml)')
    parser.add_argument('--map-path', '--mapp', metavar='PATH1:PATH2', required=False,
                        type=str, dest='map_path', default=_default_values.map_path,
                        help='Replace PATH1 in path attribute with PATH2')
    parser.add_argument('--microphone-list', '--mlist', '-m', metavar='PATH', required=True,
                        type=str, dest='microphone_list',
                        help='Microphone list file  (eg. microphone.xml)')
    parser.add_argument('--output-type', '--otype', '-t', metavar='|'.join(e.name for e in TFType), required=False, default=_default_values.output_type,
                        type=TFType, action=StoreTFTypeAction, dest='output_type',
                        help='Transfer function type  LOC=Localization,SEP=Separation,LOC_SEP=LOC&SEP (default=%(default)s)')
    parser.add_argument('--output-file', '--ofile', '-o', metavar='PATH', required=True,
                        type=str, dest='output_file',
                        help='Output transfer function zip file path  (eg. /path/to/transfer.zip)')
    parser.add_argument('--direct-length', '--dlen', '-d', metavar='NUM', required=False, default=_default_values.direct_length,
                        type=int, dest='direct_length',
                        help='Direct sound length')
    parser.add_argument('--reverb-length', '--rlen', '-r', metavar='NUM', required=False, default=_default_values.reverb_length,
                        type=int, dest='reverb_length',
                        help='Reverb sound length')
    parser.add_argument('--normalize-source', '--snorm', '-S', metavar='True|False', required=False,
                        type=bool, dest='normalize_source',
                        default=_default_values.normalize_source, help='Normalize by source')
    parser.add_argument('--normalize-microphone', '--mnorm', '-M', metavar='True|False', required=False, default=_default_values.normalize_microphone,
                        type=bool,
                        help='Normalize by microphone', dest='normalize_microphone')
    parser.add_argument('--normalize-frequency', '--fnorm', '-F', metavar='True|False', required=False, default=_default_values.normalize_frequency,
                        type=bool, dest='normalize_frequency',
                        help='Normalize by frequency')
    parser.add_argument('--reset-microphone', '--mreset', '-R', default=_default_values.reset_microphone,
                        action='store_true', dest='reset_microphone',
                        help='Remove unused channels from microphones.xml and reset their ids. (for HARKTOOL4 compatible)')
    parser.add_argument('--tsp-end', '--tend', '-e', metavar='NUM', required=False,
                        type=int, dest='tsp_end',
                        default=_default_values.tsp_end,
                        help='End point for TF calculation in sample (default TSPOffset+TSPLength)')
    parser.add_argument('--voice-channel', '--cvoice', '-c', metavar='NUM', required=False,
                        type=int, dest='voice_channel',
                        default=_default_values.voice_channel,
                        help='Channel index of the voice input)')
    parser.add_argument('--shift-length', '--slength', '-H', metavar='NUM', required=False,
                        type=int, dest='shift_length',
                        default=_default_values.shift_length,
                        help='Shift length for TF calculation (Minimum 10)')
    parser.add_argument('--head-margin', '--hmargin', '-a', metavar='NUM',
                        type=int, dest='head_margin',
                        default=_default_values.head_margin,
                        help='Head margin of impulse responces in samples')
    parser.add_argument('--log-level', '--llevel', metavar='{E|W|I|D}', required=False, default=_default_values.log_level,
                        type=LogLevel, action=StoreLogLevelAction, dest='log_level',
                        help='Log information level. (default=%(default)s)')
    parser.set_defaults(handler=main)


def main(args):
    logger = utils.initialize_logger(args)
    logger.info('calculate_tf_from_mouth_tsp start')

    calculate_tf_from_mouth_tsp(**vars(args), logger=logger)


def calculate_tf_from_mouth_tsp(
    tsp_list: str, map_path:str, microphone_list: str, output_type: str, output_file: str,
    direct_length: int, reverb_length: int,
    normalize_source: bool, normalize_microphone: bool, normalize_frequency: bool, reset_microphone: bool,
    tsp_end: int, voice_channel: int, shift_length: int, head_margin: int, logger: typing.Any, **kwargs
) -> None:

    workdir = WorkingDirectory()

    # map path
    if map_path:
        map_path_from, map_path_to = map_path.split(':')
        path_mapper = lambda path: path.replace(map_path_from, map_path_to)
    else:
        path_mapper = lambda path: path

    # read XML files
    try:
        srcFileXML: HarkXML = HarkXMLParser.from_file(tsp_list)
    except BaseException as ex:
        raise HarkToolError(
            'Failed to read impulse list file: {}'.format(tsp_list)) from ex

    try:
        micFileXML: HarkXML = HarkXMLParser.from_file(microphone_list)
    except BaseException as ex:
        raise HarkToolError(
            'Failed to read microphone list file: {}'.format(microphone_list)) from ex

    # retrieve data
    source_neighbors: Neighbors = srcFileXML.neighbors  # may be None

    source_config: Config = srcFileXML.config
    if source_config is None:
        raise HarkToolError('Config not found in source xml')

    source_positions: Positions = srcFileXML.positions
    if source_positions is None:
        raise HarkToolError('Positions not found in source xml')

    microphone_positions: Positions = micFileXML.positions
    if microphone_positions is None:
        raise HarkToolError('Positions not found in microphone xml')

    # validate parameter/data
    if voice_channel > len(microphone_positions):
        raise HarkToolError(
            '--mch shuould not less than number of microphones ({})'.format(len(microphone_positions)))

    tsp_end_limit = source_config.tsp_offset + source_config.tsp_length
    if tsp_end <= 0:
        tsp_end = tsp_end_limit
    elif tsp_end < tsp_end_limit:
        raise HarkToolError(
            '--end shuould not less than TSPOffset + TSPLength ({})'.format(tsp_end_limit))

    if head_margin > source_config.tsp_offset:
        raise HarkToolError(
            '--headmargin shuould not more than TSPOffset ({})'.format(source_config.tsp_offset))

    # sort source positions
    source_positions.sort_positions_by_id()

    # extract output channels
    output_channel_indices = select_output_channels(source_positions, microphone_positions)

    irFromMouthTSP = IRFromMouthTSP(frame_start=source_config.tsp_offset,
                                    frame_end=tsp_end,
                                    frame_head_margin=head_margin,
                                    add_white_noise=0,
                                    fft_window=source_config.n_fft,
                                    fft_hop=shift_length,
                                    sampling_rate=source_config.sampling_rate)
    extractIRVect = IRVectorExtractor(
        output_type, direct_length, reverb_length, (-1, -1), source_config.n_fft)
    extractTFs = TFExtractor(output_type=output_type,
                             normalize_source=normalize_source, normalize_microphone=normalize_microphone, normalize_frequency=normalize_frequency,
                             reset_microphone=reset_microphone)

    source_ir_vectors = []
    for source_position in source_positions.positions:
        impulse_response = irFromMouthTSP(input_sound=SoundData.from_wav(file=path_mapper(source_position.path)),
                                          input_channel=voice_channel,
                                          output_channels=output_channel_indices,
                                          workdir=workdir)
        source_ir_vectors.append(extractIRVect(impulse_response))

    loc_tfs, sep_tfs = extractTFs(source_ir_vectors)

    tf = TransferFunction(positions=source_positions, microphones=microphone_positions,
                          config=source_config, neighbors=source_neighbors, loc_tfs=loc_tfs, sep_tfs=sep_tfs)
    TransferFunctionParser.as_zipfile(tf, output_file)

# end of file
