import io
import types
import typing

from .. import utils
from ..libharkio3 import (Config, HarkXML, HarkXMLParser, Positions,
                          TransferFunction, TransferFunctionParser)
from .calctf import IRVectorExtractor, TFExtractor
from .defs import LogLevel, TFType
from .error import HarkToolError
from .utils import StoreLogLevelAction, StoreTFTypeAction
from .workset import SoundData

_default_values = types.SimpleNamespace(
    map_path=None,
    output_type=TFType.LOC_SEP,
    direct_length=32,
    reverb_length=256,
    normalize_source=False,
    normalize_microphone=False,
    normalize_frequency=False,
    reset_microphone=False,
    log_level=LogLevel.Error,
)


def setup_parser(parser):
    parser.add_argument('--impulse-list', '--ilist', '-i', metavar='PATH', required=True,
                        type=str, dest='impulse_list',
                        help='Impulse list file  (eg. impulse.xml)')
    parser.add_argument('--map-path', '--mapp', metavar='PATH1:PATH2', required=False,
                        type=str, dest='map_path', default=_default_values.map_path,
                        help='Replace PATH1 in path attribute with PATH2')
    parser.add_argument('--microphone-list', '--mlist', '-m', metavar='PATH', required=True,
                        type=str, dest='microphone_list',
                        help='Microphone list file  (eg. microphone.xml)')
    parser.add_argument('--output-type', '--otype', '-t', metavar='|'.join(e.name for e in TFType), required=False, default=_default_values.output_type,
                        type=TFType, action=StoreTFTypeAction, dest='output_type',
                        help='Transfer function type  LOC=Localization,SEP=Separation,LOC_SEP=LOC&SEP (default=%(default)s)')
    parser.add_argument('--output-file', '--ofile', '-o', metavar='PATH', required=True,
                        type=str, dest='output_file',
                        help='Output transfer function zip file path  (eg. /path/to/transfer.zip)')
    parser.add_argument('--direct-length', '--dlen', '-d', metavar='NUM', required=False, default=_default_values.direct_length,
                        type=int, dest='direct_length',
                        help='Direct sound length')
    parser.add_argument('--reverb-length', '--rlen', '-r', metavar='NUM', required=False, default=_default_values.reverb_length,
                        type=int, dest='reverb_length',
                        help='Reverb sound length')
    parser.add_argument('--normalize-source', '--snorm', '-S', metavar='True|False', required=False,
                        type=bool, dest='normalize_source',
                        default=_default_values.normalize_source, help='Normalize by source')
    parser.add_argument('--normalize-microphone', '--mnorm', '-M', metavar='True|False', required=False, default=_default_values.normalize_microphone,
                        type=bool,
                        help='Normalize by microphone', dest='normalize_microphone')
    parser.add_argument('--normalize-frequency', '--fnorm', '-F', metavar='True|False', required=False, default=_default_values.normalize_frequency,
                        type=bool, dest='normalize_frequency',
                        help='Normalize by frequency')
    parser.add_argument('--reset-microphone', '--mreset', '-R', default=_default_values.reset_microphone,
                        action='store_true', dest='reset_microphone',
                        help='Remove unused channels from microphones.xml and reset their ids. (for HARKTOOL4 compatible)')
    parser.add_argument('--log-level', '--llevel', metavar='{E|W|I|D}', required=False, default=_default_values.log_level,
                        type=LogLevel, action=StoreLogLevelAction, dest='log_level',
                        help='Log information level. (default=%(default)s)')
    parser.set_defaults(handler=main)


def main(args):
    logger = utils.initialize_logger(args)
    logger.info('harktool_calctfimpExec start')

    calculate_tf_from_impulse_response(**vars(args), logger=logger)


def calculate_tf_from_impulse_response(
        impulse_list: str, map_path:str, microphone_list: str, output_type: str, output_file: str,
        direct_length: int, reverb_length: int,
        normalize_source: bool, normalize_microphone: bool, normalize_frequency: bool, reset_microphone: bool,
        logger: typing.Any, **kwargs) -> None:
    
    # for README
    message_buffer: io.IOBase = io.StringIO()

    # map path
    if map_path:
        map_path_from, map_path_to = map_path.split(':')
        path_mapper = lambda path: path.replace(map_path_from, map_path_to)
    else:
        path_mapper = lambda path: path

    # read XML files
    try:
        source_xml: HarkXML = HarkXMLParser.from_file(impulse_list)
        logger.info(f'Read impulse list file: {impulse_list}')
    except BaseException as ex:
        raise HarkToolError(
            f'Failed to read impulse list file: {impulse_list}') from ex

    try:
        microphone_xml: HarkXML = HarkXMLParser.from_file(microphone_list)
        logger.info(f'Read microphone list file: {microphone_list}')
    except BaseException as ex:
        raise HarkToolError(
            f'Failed to read microphone list file: {microphone_list}') from ex

    # retrieve data
    source_config: Config = source_xml.config
    if source_config is None:
        raise HarkToolError('Config not found in source xml')

    source_positions: Positions = source_xml.positions
    if source_positions is None:
        raise HarkToolError('Positions not found in source xml')

    microphone_positions: Positions = microphone_xml.positions
    if microphone_positions is None:
        raise HarkToolError('Positions not found in microphone xml')

    # sort source positions
    source_positions.sort_positions_by_id()

    # setup worker instance
    extract_irvector = IRVectorExtractor(
        output_type, direct_length, reverb_length, (-1, -1), source_config.n_fft, logger=logger, message_buffer=message_buffer)
    extract_tfs = TFExtractor(
        output_type=output_type,
        normalize_source=normalize_source, normalize_microphone=normalize_microphone, normalize_frequency=normalize_frequency,
        reset_microphone=reset_microphone,
        logger=logger, message_buffer=message_buffer)

    # tf calculation
    source_ir_vectors = [extract_irvector(impulse_responses=SoundData.from_wav(
        path_mapper(position.path))) for position in source_positions.positions]
    loc_tfs, sep_tfs = extract_tfs(source_ir_vectors)

    # save as zip file
    tf = TransferFunction(positions=source_positions, microphones=microphone_positions,
                          config=source_config, neighbors=None, loc_tfs=loc_tfs, sep_tfs=sep_tfs)
    TransferFunctionParser.as_zipfile(tf, output_file)

# end of file
