#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/app/calctfimp.py

"""

import types
from typing import Any

from .. import utils
from .calctf import IRFromTSP
from .defs import LogLevel, SoundType
from .error import HarkToolError
from .utils import StoreLogLevelAction
from .workset import SoundData

_default_values = types.SimpleNamespace(
    original_tsp_channel=0,
    signal_maximum=0,
    samples_by_frame=16384,
    synchronous_addition_frequency=16,
    synchronous_addition_offset=16384,
    search_offset=(-1, -1),
    output_channels=[],
    channel_origin=0,
    log_level=LogLevel.Error,
)


def setup_parser(parser):
    """calcsysimp 機能で使用する ArgumentParser の設定を行う。

    Args:
        parser (argparse.ArgumentParser): calcsysimp 機能で使用する ArgumentParser インスタンス
    """
    parser.add_argument('--original-tsp-raw-file', '--tspf', '-F', metavar='PATH', required=False,
                        type=str, dest='original_tsp_raw_file',
                        help='Original tsp filename (float-data format)')
    parser.add_argument('--original-tsp-wav-file', '--tspw', '-W', metavar='PATH', required=False,
                        type=str, dest='original_tsp_wav_file',
                        help='Original tsp filename (PCM-wave format)')
    parser.add_argument('--original-tsp-channal', '--tspc', '-r', metavar='NUM', required=False,
                        type=int, dest='original_tsp_channel',
                        default=_default_values.original_tsp_channel,
                        help='Channel number (for the PCM-wave tsp) (default=%(default)s)')
    parser.add_argument('--signal-maximum', '--smax', '-m', metavar='NUM', required=False,
                        type=int, dest='signal_maximum',
                        default=_default_values.signal_maximum,
                        help='Maximum signal value (for float-format tsp) (default=%(default)s)')
    parser.add_argument('--samples-by-frame', '--fsample', '-n', metavar='NUM', required=False,
                        type=int, dest='samples_by_frame',
                        default=_default_values.samples_by_frame,
                        help='Number of samplings per frame  (default=%(default)s)')
    parser.add_argument('--synchronous-addition-frequency', '--fsync', '-f', metavar='NUM', required=False,
                        type=int, dest='synchronous_addition_frequency',
                        default=_default_values.synchronous_addition_frequency,
                        help='Frequency of synchronous addition (default=%(default)s)')
    parser.add_argument('--synchronous-addition-offset', '--osync', '-a', metavar='NUM', required=False,
                        type=int, dest='synchronous_addition_offset',
                        default=_default_values.synchronous_addition_offset,
                        help='Start position of synchronous addition (default=%(default)s)')
    parser.add_argument('--search-offset', '--soffset', '-s', nargs=2, metavar='NUM', required=False,
                        type=int,  dest='search_offset',
                        default=_default_values.search_offset,
                        help='Start and end point of the search range. This enables to search a start position and use it.')
    parser.add_argument('--output-channel-list', '--coutput', '-l', nargs='+', metavar='NUM', required=False,
                        type=int, dest='output_channels',
                        default=_default_values.output_channels,
                        help='Select channel(s) to calculate. (zero-based)')
    parser.add_argument('--input-wav-file', '--iwave', '-i', metavar='PATH', required=True,
                        type=str, dest='input_wav_file',
                        help='Recorded wav file')
    parser.add_argument('--output-raw-file', '--oraw', '-o', metavar='PATH', required=True,
                        type=str, dest='output_raw_file',
                        help='Path prefix for output impulse response files. Output file will be 32bit float monaural raw data named <PATH>_ch0.flt, <PATH>_ch1.flt, ...')
    parser.add_argument('--channel-origin', '--corigin', '-g', metavar='NUM', required=False,
                        type=int, dest='channel_origin',
                        default=_default_values.channel_origin,
                        help='Number of first channel. This is used as the channel number of output file name. (default=%(default)s)')
    parser.add_argument('--log-level', '--llevel', metavar='{E|W|I|D}', required=False,
                        type=LogLevel, action=StoreLogLevelAction, dest='log_level',
                        default=_default_values.log_level,
                        help='Log information level. (default=%(default)s)', )
    parser.set_defaults(handler=main)


def main(args):
    """calcsysimp 機能のメイン関数

    Args:
        args (list[str]): calcsysimp 機能の起動オプションを収めた list

    Raises:
        HarkToolError: 起動時のオプション指定が正しくない場合に raise される
    """    
    logger = utils.initialize_logger(args)
    logger.info('harktool_calctfimpExec start')

    # validation
    if args.original_tsp_raw_file and args.original_tsp_raw_file:
        raise HarkToolError('Duplicate --original-tsp-raw-file or --original-tsp-wav-file')
    if not args.original_tsp_raw_file and not args.original_tsp_raw_file:
        raise HarkToolError('Missing --original-tsp-raw-file or --original-tsp-wav-file')
    if args.original_tsp_raw_file:
        args.original_tsp_file = args.original_tsp_raw_file
        args.original_tsp_type = SoundType.RAW
    if args.original_tsp_wav_file:
        args.original_tsp_file = args.original_tsp_wav_file
        args.original_tsp_type = SoundType.WAV

    calculate_impulse_response_from_tsp(**vars(args), logger=logger)


def calculate_impulse_response_from_tsp(
    original_tsp_file: str,
    original_tsp_type: SoundType,
    original_tsp_channel: int,
    samples_by_frame: int,
    synchronous_addition_frequency: int,
    synchronous_addition_offset: int,
    signal_maximum: int,
    output_channels: list[int],
    search_offset: int,
    input_wav_file: str,
    output_raw_file: str,
    logger: Any,
):
    """HARKTOOL5 における、calcsysimp 処理の本体

    Args:
        original_tsp_file (str): TSPファイル。original_tsp_type に応じた形式のファイルのパスを指定する。
        original_tsp_type (SoundType): TSPファイルの種類。SoundType.RAW または SoundType.WAV を指定する。
        original_tsp_channel (int): _description_
        samples_by_frame (int): フレームあたりのサンプル数を指定する。
        synchronous_addition_frequency (int): _description_
        synchronous_addition_offset (int): _description_
        signal_maximum (int): _description_
        output_channels (list[int]): _description_
        search_offset (int): _description_
        input_wav_file (str): _description_
        output_raw_file (str): _description_
        logger (Any): _description_

    Raises:
        HarkToolError: _description_
    """
    # main process
    logger.info('harktool_calcsysimpExec start')

    # read tsp file
    if original_tsp_type == SoundType.RAW:
        original_tsp = SoundData.from_raw(original_tsp_file, sampling_rate=16000)
    elif original_tsp_type == SoundType.WAV:
        original_tsp = SoundData.from_wav(original_tsp_file)
    else:
        raise HarkToolError(f'Unexpected original tsp type: {original_tsp_type}')

    # setup calculation
    ir_from_tsp = IRFromTSP(
        sampling_rate=original_tsp.sampling_rate,
        samples_by_frame=samples_by_frame,
        sync_add_frequency=synchronous_addition_frequency, 
        sync_add_offset=synchronous_addition_offset,
        signal_maximum=signal_maximum)

    # calculate impulse response
    impulse_response = ir_from_tsp(
        tsp_original_signal=original_tsp,
        tsp_original_channel=original_tsp_channel,
        input_signal=SoundData.from_wav(file=input_wav_file),
        output_channels=output_channels, 
        search_offset=search_offset)

    # save as file
    impulse_response.as_file(output_raw_file)

# end of file
