#! /usr/bin/env python3

# An interface that supports file input and output in ndjson format.
# See this URL for the specifications.
#   https://github.com/ndjson/ndjson-spec
# Support for other languages appears to be available here.
#   https://github.com/ndjson
#
# There is a public module "ndjson" on PyPI, but we do not use it.
# We implement our own interface to customize file I/O and conversion in PyHARK.
# If you want to use this special encoder and decoder directly from outside,
# use the following.
# - JSON encoder/decoder with PyHARK types
#   * Encoder : ExJSONEncoder
#   * Decoder : ExJSONDecoder
# - NDJSON encoder/decoder with PyHARK types
#   * Encoder : NDJSONEncoder
#   * Decoder : NDJSONDecoder

import os
import json
import numpy
import time
import datetime


# The frames class is just an alias for the list class, but within this module
# it has the following meaning:
# When writing or reading ndjson files, the encoder and decoder use the frames class
# to determine whether the passed object is a single frame or multiple frames.
class frames(list):
    pass

# ExJSONEncoder and ExJSONDecoder support types that the json module does not
# support (e.g. types supported by numpy, complex numbers, time, etc.) and
# HARK-specific types (classes implemented in C++, such as Source, TimeStamp,
# TransferFunction, etc.).
# When ExJSONEncoder is called directly, it will determine whether obj
# is an instance of frames so that it can be handled without being aware
# of the difference with NDJSONEncoder, but please note that when called
# from NDJSONEncoder, an instance of frames is never passed.
class ExJSONEncoder(json.JSONEncoder):
    def default(self, obj):
        _debug = True
        if isinstance(obj, frames):
            if _debug: print("[frames]")
            return list(obj)
        elif isinstance(obj, complex):
            if _debug: print("[complex]")
            return {
                "type": "complex",
                "data": [float(obj.real), float(obj.imag)],
            }
        elif isinstance(obj, datetime.datetime):
            if _debug: print("[datetime.datetime]")
            return {
                "type": "datetime.datetime",
                "isoformat": obj.isoformat()
            }
        elif isinstance(obj, datetime.timedelta):
            if _debug: print("[datetime.timedelta]")
            return {
                "type": "datetime.timedelta",
                "days": obj.days,
                "seconds": obj.seconds,
                "microseconds": obj.microseconds
            }
        # elif isinstance(obj, numpy.complex_): # Obsoleted in numpy 2
        elif isinstance(obj, (numpy.complex64, numpy.complex128, numpy.complex256)):
            if _debug: print("[numpy.complex]")
            return {
                "type": "numpy.complex",
                "data": obj.astype(str),
                "shape": obj.shape,
                "dtype": str(obj.dtype)
            }
        elif isinstance(obj, (numpy.int_, numpy.intc, numpy.intp, numpy.int8, numpy.int16, numpy.int32, numpy.int64, numpy.uint8, numpy.uint16, numpy.uint32, numpy.uint64)):
            if _debug: print("[numpy.int]")
            return {
                "type": "numpy.int",
                "data": obj.astype(str),
                "shape": obj.shape,
                "dtype": str(obj.dtype)
            }
        # elif isinstance(obj, numpy.float_): # Obsoleted in numpy 2
        elif isinstance(obj, (numpy.float16, numpy.float32, numpy.float64, numpy.float128)):
            if _debug: print("[numpy.float]")
            return {
                "type": "numpy.float",
                "data": obj.astype(str),
                "shape": obj.shape,
                "dtype": str(obj.dtype)
            }
        elif isinstance(obj, numpy.bool_):
            if _debug: print("[numpy.bool]")
            return {
                "type": "numpy.bool",
                "data": obj.astype(str),
                "shape": obj.shape,
                "dtype": str(obj.dtype)
            }
        elif isinstance(obj, numpy.datetime64):
            if _debug: print("[numpy.datetime]")
            return {
                "type": "numpy.datetime",
                "data": obj.astype(str),
                "shape": obj.shape,
                "dtype": str(obj.dtype)
            }
        elif isinstance(obj, numpy.datetime64):
            if _debug: print("[numpy.timedelta]")
            # This days, seconds and microseconds elements are datetime
            # module compatible, so the maximum for seconds is not 60
            # but the total number of seconds in a day, which is 60*60*24.
            return {
                "type": "numpy.timedelta",
                "days": obj.astype('timedelta64[D]').astype(int),
                "seconds": obj.astype('timedelta64[s]').astype(int) % (60*60*24),
                "microseconds": obj.astype('timedelta64[us]').astype(int) % (1000000),
                "shape": obj.shape,
                "dtype": str(obj.dtype)
            }
        elif isinstance(obj, numpy.ndarray):
           if _debug: print("[numpy.ndarray]")
           return {
               "type": "numpy.ndarray",
               "data": obj.tolist(),
               "shape": obj.shape,
               "dtype": str(obj.dtype)
           }
        #return json.JSONEncoder.default(self, obj)
        return super(ExJSONEncoder, self).default(self, obj)

class ExJSONDecoder(json.JSONDecoder):
    def __init__(self, *args, **kwargs):
        # This is a dictionary of type classes that support conversion between types and strings in numpy.
        # That is, you can do "numpy.complex(numpy.complex(1+2j).astype(str))".
        # Therefore, numpy.timedelta64 type is not included because
        # "numpy.timedelta64(numpy.timedelta64().astype(str))" is not feasible.
        self.npstrtype = {
            "numpy.complex": {
                "complex64": numpy.complex64,
                "complex128": numpy.complex128,
                "complex256": numpy.complex256,
                "default": numpy.complex128
            },
            "numpy.int": {
                "int8": numpy.int8,
                "int16": numpy.int16,
                "int32": numpy.int32,
                "int64": numpy.int64,
                "uint8": numpy.uint8,
                "uint16": numpy.uint16,
                "uint32": numpy.uint32,
                "uint64": numpy.uint64,
                "default": numpy.int64
            },
            "numpy.float": {
                "float16": numpy.float16,
                "float32": numpy.float32,
                "float64": numpy.float64,
                "float128": numpy.float128,
                "default": numpy.float64
            },
            "numpy.bool": {
                "bool": numpy.bool_,
                "default": numpy.bool_
            },
            "numpy.datetime": {
                "default": numpy.datetime64
            }
        }
        json.JSONDecoder.__init__(self, object_hook=self.object_hook, *args, **kwargs)

    def object_hook(self, obj):
        if isinstance(obj, dict) and "type" in obj and isinstance(obj["type"], str):
            if obj["type"] == "complex":
                if ("data" not in obj) or (not isinstance(obj["data"], list)):
                    return complex(0, 0)
                return complex(
                    0 if len(obj["data"]) < 1 else float(obj["data"][0]),
                    0 if len(obj["data"]) < 2 else float(obj["data"][1])
                )
            elif obj["type"].startswith("datetime."):
                if obj["type"] == "datetime.datetime":
                    if ("isoformat" not in obj) or (not isinstance(obj["isoformat"], str)):
                        return datetime.datetime.min
                    return datetime.datetime.fromisoformat(obj["isoformat"])
                elif obj["type"] == "datetime.timedelta":
                    return datetime.timedelta(
                        days=0 if "days" not in obj else int(obj["days"]),
                        seconds=0 if "seconds" not in obj else int(obj["seconds"]),
                        microseconds=0 if "microseconds" not in obj else int(obj["microseconds"])
                    )
            elif obj["type"].startswith("numpy."):
                if "dtype" in obj and isinstance(obj["dtype"], str):
                    if obj["type"] in self.npstrtype.keys():
                        dinst = self.npstrtype[obj["type"]]["default"]
                        if obj["dtype"] in self.npstrtype[obj["type"]].keys():
                            disnt = self.npstrtype[obj["type"]][obj["dtype"]]
                        if isinstance(obj["data"], str):
                            return dinst(obj["data"])
                        else:
                            return dinst()
                    elif obj["type"] in ["numpy.timedelta"]:
                        pass
                    elif obj["type"] in ["numpy.ndarray"]:
                        pass
            elif obj["type"].startswith("hark."):
                pass
        return obj
        

# This class inherits from ExJSONEncoder and reproduces the results of writing
# per frame when frames (not to be confused with a list) are input.
# Writing a single frame produces the same results as ExJSONEncoder.
# Note that no newline character is added to the last element.
class NDJSONEncoder(ExJSONEncoder):
    def encode(self, obj, *args, **kwargs):
        if isinstance(obj, frames):
            return "\n".join([super(NDJSONEncoder, self).encode(element, *args, **kwargs) for element in obj])
        return super(NDJSONEncoder, self).encode(obj, *args, **kwargs)

# Note that the final character of the ndjson string (ndjstr) is not a newline character.
class NDJSONDecoder(ExJSONDecoder):
    def decode(self, ndjstr, *args, **kwargs):
        return super(NDJONDecoder, self).decode("[{}]".format(", ".join(ndjstr.split('\n'))), *args, **kwargs)

class IFndjson(object):
    def __init__(self, filepath, **kwargs):
        self.data = []
        self.f = None
        self.rw = ""

        self.filepath = filepath
        self.coversion = False
        if os.path.splitext(filepath)[1] == ".json":
            # Treat json file as ndjson file.
            # e.g.) ndjson file format is used to write earch frame and
            #     then converted to json file format after filalization.
            #     Reading the file simply acts as an auto-detection of
            #     the format.
            self.conversion = True
        elif os.path.splitext(filepath)[1] == ".ndjson":
            self.conversion = False
        else:
            # Checks the file format when loading a file.
            # This mode definitely slows down the processing speed.
            self.conversion = False

    def __del__(self):
        if self.data and (self.rw == "w"):
            self.write()


    # The dump(), dumps(), load() and loads() member methods for ndjson
    # in this class are compatible with the json module. 
    def dump(self, obj, fp, **kwargs):
        kwargs["cls"] = NDJSONEncoder
        fp.write(NDJSONEncoder(**kwargs).encode(obj))

    def dumps(self, *args, **kwargs):
        kwargs["cls"] = NDJSONEncoder
        return json.dumps(*args, **kwargs)

    def load(self, *args, **kwargs):
        kwargs["cls"] = NDJSONDecoder
        return json.load(*args, **kwargs)

    def loads(self, *args, **kwargs):
        kwargs["cls"] = NDJSONDecoder
        return json.loads(*args, **kwargs)

    def exists():
        if self.filepath:
            return os.path.isfile(self.filepath)
        else:
            print("[Error] No filepath has been specified yet.")
            return False

    def open(self, rw="r"):
        if self.f is not None:
            self.close()
            self.f = None
        if rw == "r":
            if not self.exists():
                raise FileNotFoundError("[Error] IFndjson.py open(): File not found: {}".format(self.filepath))
            self.f = open(self.filepath, mode="r", newline="\n")
            self.rw = "r"
        elif rw == "a":
            if not self.exists():
                print("[Warning] Append mode is selected, but the file to append to was not found so a new file will be created.")
            self.f = open(self.filepath, "a")
            self.rw = "w"
        elif rw == "w":
            if self.exists():
                print("[Warning] The destination file already exists, but the contents have been overwritten because the file mode is not append mode.")
            self.f = open(self.filepath, mode="w", newline="\n")
            self.rw = "w"
        else:
            print("[Info] The file was opened in read mode because the mode was not specified correctly. Supported modes are: 'r' : read mode (default i.e. not specified was included), 'a': append mode, 'w': write mode (overwrite mode).")
            if not self.exists():
                raise FileNotFoundError("[Error] IFndjson.py open(): File not found: {}".format(self.filepath))
            self.f = open(self.filepath, mode="r", newline="\n")
            self.rw = "r"

    def close(self):
        if self.f:
            self.f.close()
            self.f = None
            self.rw = ""

    def write(self, obj, is_single=True, lazily=True):
        single = is_single
        if not is_single and not isinstance(obj, list):
            # If obj is not a list then it assumes obj is a single frame,
            # because obj being multi-frame means obj is a list of frames.
            single = True
        if lazily:
            # Every frame is written out at the destructor time.
            if single:
                self.data.append(obj)
            else:
                self.data.extend(obj)
        elif self.conversion:
            # This mode requires the file to be reopened for each write time,
            # so if it is already open in an instance, it will be forcibly closed.
            if self.f is not None:
                self.close()
            # The json structure is not suitable for appending, so if you
            # want to write a frame every time you should choose the ndjson
            # format. Or you should do the bulk write in the destructor.
            # In other words, in this mode it reads already written frames,
            # appends the frames, and writes them back.
            _data = []
            if os.path.isfile(self.filepath):
                with open(self.filepath, mode="r", newline="\n") as _f:
                    _data = json.load(_f)
            if self.data:
                # Add the pending write data to the write data.
                _data.extend(self.data)
                self.data = []
            if single:
                _data.append(obj)
            else:
                _data.extend(obj)
            with open(self.filepath, mode="w", newline="\n") as _f:
                _f.write(json.dumps(_data))
        else:
            # If you are selected file format the ndjson format in this class,
            # keep the file open in an instance to avoid the overhead of
            # opening and closing the file.
            if (self.f is None) or (self.rw == "r"):
                if self.exists():
                    self.open(rw="a")
                else:
                    self.open(rw="w")
            if self.data:
                # Writes any pending write data to a file.
                self.f.write(self.dumps(self.data, **self.kwargs) + "\n")
                self.data = []
            self.f.write(self.dumps(obj, **self.kwargs) + "\n")

    def read(self, tryread=0, do_bulkloading=False):
        # The json structure does not offer any option other than
        # loading the list in bulk, if you need to load many frames,
        # choose the ndjson file format which allows loading frame by frame.
        bulkloading = True if self.conversion else do_bulkloading
        if bulkloading:
            if self.rw != "r":
                # If the file is not already open or is already open in write mode,
                # it is reopened in read mode.
                self.open(rw="r")
                self.data = []
                if self.conversion:
                    self.data = json.load(self.f)
                else:
                    self.data = self.load(self.f)
                # Do not close the file now, even if you are done reading it.
                # The file will be freed automatically in the destructor.
                # If you want to loop reading a file, you must explicitly call
                # close() to signal the file to be read again.
            if self.data:
                # If tryread is 0 then it returns a single frame, otherwise
                # it attempts to read up to tryread count and returns them
                # as a frame list, or None if all data has been read.
                if tryread == 0:
                    return self.data.pop(0)
                else:
                    ret = []
                    for i in range(tryread):
                        if self.data:
                            ret.append(self.data.pop(0))
                    return ret
            return None
        else:
            if self.rw != "r":
                self.open(rw="r")
            framestr = ""
            ret = None
            not_eof = True
            if tryread == 0:
                while not_eof:
                    try:
                        framestr = next(self.f).strip()
                        if framestr:
                            ret = self.loads(framestr, **kwargs)
                    except StopIteration:
                        print("EOF")
                        not_eof = False
                        break
            else:
                for i in range(tryread):
                    while not_eof:
                        try:
                            framestr = next(self.f).strip()
                            if framestr:
                                # If not even a single frame can be obtained,
                                # it will return None, so it will be initialized
                                # as a list type the first time a value is obtained.
                                if ret is None:
                                    ret = []
                                ret.append(self.loads(framestr, **kwargs))
                        except StopIteration:
                            print("EOF")
                            not_eof = False
                            break
        return ret

