#! /usr/bin/env python3
# -*- coding: utf-8 -*-

from decimal import Decimal
from math import ceil, floor, log10

import kivy
import numpy
from kivy import metrics
from kivy.app import App
from kivy.clock import Clock
from kivy.config import Config
from kivy.core.window import Window
from kivy.graphics import Color, Fbo, Line, Mesh, Rectangle, RenderContext
from kivy.graphics.texture import Texture
from kivy.lang import Builder
from kivy.properties import (
    BooleanProperty,
    BoundedNumericProperty,
    DictProperty,
    ListProperty,
    NumericProperty,
    ObjectProperty,
    StringProperty,
)
from kivy.uix.boxlayout import BoxLayout
from kivy.uix.button import Button
from kivy.uix.image import Image
from kivy.uix.label import Label
from kivy.uix.stencilview import StencilView
from kivy.uix.widget import Widget
from kivy.utils import get_color_from_hex as rgb
from kivy_garden.graph import BarPlot, Graph, LinePlot, Plot

kivy.require("1.8.0")
Config.set("kivy", "log_level", "error")
Config.set("kivy", "log_enable", "0")
Config.write()


class HBarPlot(Plot):
    """BarPlot class which displays a bar graph."""

    bar_height = NumericProperty(1)
    bar_spacing = NumericProperty(1.0)
    graph = ObjectProperty(allownone=True)

    def __init__(self, *ar, **kw):
        super().__init__(*ar, **kw)
        self.bind(bar_height=self.ask_draw)
        self.bind(points=self.update_bar_height)
        #        self.bind(graph=self.update_bar_height)
        self.range = None

    def set_cursor_range(self, min, max):
        if min is None:
            self.range = None
        else:
            self.range = (min, max)

    def update_bar_height(self, *ar):
        if not self.graph:
            return
        if len(self.points) < 2:
            return
        if self.graph.ymax == self.graph.ymin:
            return

        point_height = (
            len(self.points)
            * float(abs(self.graph.ymax) + abs(self.graph.ymin))
            / float(abs(max(self.points)[1]) + abs(min(self.points)[1]))
        )

        if not self.points:
            self.bar_height = 1
        else:
            self.bar_height = (
                (self.graph.height - self.graph.padding)
                / point_height
                * self.bar_spacing
            )

    def create_drawings(self):
        self._backcolor = Color(0.25, 0.75, 0.75, 1.0)
        self._color = Color(*self.color)
        self._backmesh = Rectangle(pos=(0, 0), size=(100, 100))
        self._mesh = Mesh()
        self.bind(color=lambda instr, value: setattr(self._color, "rgba", value))
        return (self._backcolor, self._backmesh, self._color, self._mesh)

    def draw(self, *args):
        super().draw(*args)
        points = self.points

        # The mesh only supports (2^16) - 1 indices, so...
        if len(points) * 6 > 65535:
            # Logger.error(
            #     "HBarPlot: cannot support more than 10922 points. "
            #     "Ignoring extra points."
            # )
            print(
                "HBarPlot: cannot support more than 10922 points. "
                "Ignoring extra points."
            )
            points = points[:10922]

        bounds = self.get_px_bounds()
        if self.range is None:
            self._backmesh.pos = (0, 0)
            self._backmesh.size = (0, 0)
        else:
            self._backmesh.pos = (0, self.range[0])
            self._backmesh.size = (
                bounds["xmax"] - bounds["xmin"],
                self.range[1] - self.range[0],
            )

        point_len = len(points)
        mesh = self._mesh
        mesh.mode = "triangles"
        vert = mesh.vertices
        ind = mesh.indices
        diff = len(points) * 6 - len(vert) // 4
        if diff < 0:
            del vert[4 * point_len :]
            del ind[point_len:]
        elif diff > 0:
            ind.extend(range(len(ind), len(ind) + diff))
            vert.extend([0] * (diff * 4))

        x_px = self.x_px()
        y_px = self.y_px()
        #        xmin = x_px(0)
        #        xmax = bounds["xmax"]
        #        print('bounds: {}'.format(bounds))

        #        print('params: {}'.format(self.params))

        bar_height = self.bar_height
        if bar_height < 0:
            bar_height = y_px(bar_height) - bounds["ymin"]

        for k, pp in enumerate(self.iterate_points()):
            # p  = points[k]
            x1 = pp[0]
            x2 = x_px(0)
            y1 = pp[1]
            y2 = y1 + bar_height  # bar_height

            idx = k * 24
            # first triangle
            vert[idx] = x1
            vert[idx + 1] = y2
            vert[idx + 4] = x1
            vert[idx + 5] = y1
            vert[idx + 8] = x2
            vert[idx + 9] = y1
            # second triangle
            vert[idx + 12] = x1
            vert[idx + 13] = y2
            vert[idx + 16] = x2
            vert[idx + 17] = y2
            vert[idx + 20] = x2
            vert[idx + 21] = y1
        mesh.vertices = vert

    def _unbind_graph(self, graph):
        graph.unbind(
            height=self.update_bar_height,
            xmin=self.update_bar_height,
            ymin=self.update_bar_height,
        )

    def bind_to_graph(self, graph):
        old_graph = self.graph

        if old_graph:
            # unbind from the old one
            self._unbind_graph(old_graph)

        # bind to the new one
        self.graph = graph
        graph.bind(
            height=self.update_bar_height,
            xmin=self.update_bar_height,
            ymin=self.update_bar_height,
        )

    def unbind_from_graph(self):
        if self.graph:
            self._unbind_graph(self.graph)


Builder.load_string(
    """
<GraphRotatedLabel>:
    canvas.before:
        PushMatrix
        Rotate:
            angle: root.angle
            axis: 0, 0, 1
            origin: root.center
    canvas.after:
        PopMatrix
"""
)


class GraphRotatedLabel(Label):
    angle = NumericProperty(0)


def identity(x):
    return x


def exp10(x):
    return 10 ** x


# スペクトログラムとヒストグラムなどを一緒に表示するグラフ
class KivyImageGraph(Widget):
    """Graph class, see module documentation for more information."""

    # triggers a full reload of graphics
    _trigger = ObjectProperty(None)
    # triggers only a repositioning of objects due to size/pos updates
    _trigger_size = ObjectProperty(None)
    # triggers only a update of colors, e.g. tick_color
    _trigger_color = ObjectProperty(None)
    # holds widget with the x-axis label
    _xlabel = ObjectProperty(None)
    # holds widget with the y-axis label
    _ylabel = ObjectProperty(None)
    # holds widget with the f-axis label
    _flabel = ObjectProperty(None)
    # holds widget with the p-axis label
    _plabel = ObjectProperty(None)
    # holds all the x-axis tick mark labels
    _x_grid_label = ListProperty([])
    # holds all the y-axis tick mark labels
    _y_grid_label = ListProperty([])
    # holds all the f-axis tick mark labels
    _f_grid_label = ListProperty([])
    # holds all the p-axis tick mark labels
    _p_grid_label = ListProperty([])
    # the mesh drawing all the ticks/grids
    _mesh_ticks = ObjectProperty(None)
    # the mesh which draws the surrounding rectangle
    _waveform_mesh_rect = ObjectProperty(None)
    _spectrogram_mesh_rect = ObjectProperty(None)
    _histogram_mesh_rect = ObjectProperty(None)
    _valid_power_area_mesh = ObjectProperty(None)
    _valid_freq_area_mesh = ObjectProperty(None)
    # a list of locations of major and minor ticks. The values are not
    # but is in the axis min - max range
    _ticks_majorx = ListProperty([])
    _ticks_minorx = ListProperty([])
    _ticks_majory = ListProperty([])
    _ticks_minory = ListProperty([])
    _ticks_majorf = ListProperty([])
    _ticks_minorf = ListProperty([])
    _ticks_majorp = ListProperty([])
    _ticks_minorp = ListProperty([])

    valid_power_range = ListProperty([None, None])
    valid_freq_range = ListProperty([None, None])

    drag_origin = ()
    drag_freq = False
    drag_power = False

    tick_color = ListProperty([0.25, 0.25, 0.25, 1])
    """Color of the grid/ticks, default to 1/4. grey.
    """

    background_color = ListProperty([0, 0, 0, 0])
    """Color of the background, defaults to transparent
    """

    border_color = ListProperty([1, 1, 1, 1])
    """Color of the border, defaults to white
    """

    valid_area_color = ListProperty([0.25, 0.5, 0.5, 1])

    label_options = DictProperty()
    """Label options that will be passed to `:class:`kivy.uix.Label`.
    """

    _with_stencilbuffer = BooleanProperty(True)
    """Whether :class:`Graph`'s FBO should use FrameBuffer (True) or not (False).

    .. warning:: This property is internal and so should be used with care. It can break
    some other graphic instructions used by the :class:`Graph`, for example you can have
    problems when drawing :class:`SmoothLinePlot` plots, so use it only when you know
    what exactly you are doing.

    :data:`_with_stencilbuffer` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to True.
    """

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        with self.canvas:
            self._fbo = Fbo(size=self.size, with_stencilbuffer=self._with_stencilbuffer)

        with self._fbo:
            self._background_color = Color(*self.background_color)
            self._background_rect = Rectangle(size=self.size)
            self._mesh_ticks_color = Color(*self.tick_color)
            self._mesh_ticks = Mesh(mode="lines")
            self._mesh_rect_color = Color(*self.border_color)
            self._waveform_mesh_rect = Mesh(mode="line_strip")
            self._spectrogram_mesh_rect = Mesh(mode="line_strip")
            self._histogram_mesh_rect = Mesh(mode="line_strip")
            self._valid_power_area_color = Color(*self.valid_area_color)
            self._valid_power_area_mesh = Mesh(mode="triangle")
            self._valid_freq_area_color = Color(*self.valid_area_color)
            self._valid_freq_area_mesh = Mesh(mode="triangle")

        with self.canvas:
            Color(1, 1, 1)
            self._fbo_rect = Rectangle(size=self.size, texture=self._fbo.texture)

        for mesh in (
            self._waveform_mesh_rect,
            self._spectrogram_mesh_rect,
            self._histogram_mesh_rect,
            self._valid_power_area_mesh,
            self._valid_freq_area_mesh,
        ):
            mesh.vertices = [0] * (5 * 4)
            mesh.indices = range(5)

        for mesh in (self._valid_power_area_mesh, self._valid_freq_area_mesh):
            mesh.indices = [0, 1, 2, 0, 2, 3]

        self._plot_area = StencilView()
        self._spectrogram_area = Image(fit_mode="fill")
        self._power_histogram_area = StencilView()
        self._power_colorbar_area = Widget()
        self._power_colorbar_texture = None

        self.add_widget(self._plot_area)
        self.add_widget(self._spectrogram_area)
        self.add_widget(self._power_histogram_area)
        self.add_widget(self._power_colorbar_area)

        t = self._trigger = Clock.create_trigger(self._redraw_all)
        ts = self._trigger_size = Clock.create_trigger(self._redraw_size)
        tc = self._trigger_color = Clock.create_trigger(self._update_colors)

        self.bind(
            center=ts,
            padding=ts,
            precision=ts,
            plots=ts,
            x_grid=ts,
            y_grid=ts,
            f_grid=ts,
            p_grid=ts,
            draw_border=ts,
            valid_power_range=ts,
            valid_freq_range=ts,
        )
        self.bind(
            xmin=t,
            xmax=t,
            xlog=t,
            x_ticks_major=t,
            x_ticks_minor=t,
            xlabel=t,
            x_grid_label=t,
            ymin=t,
            ymax=t,
            ylog=t,
            y_ticks_major=t,
            y_ticks_minor=t,
            ylabel=t,
            y_grid_label=t,
            fmin=t,
            fmax=t,
            flog=t,
            f_ticks_major=t,
            f_ticks_minor=t,
            flabel=t,
            f_grid_label=t,
            pmin=t,
            pmax=t,
            plog=t,
            p_ticks_major=t,
            p_ticks_minor=t,
            plabel=t,
            p_grid_label=t,
            font_size=t,
            label_options=t,
            x_ticks_angle=t,
        )
        self.bind(tick_color=tc, background_color=tc, border_color=tc)
        self._trigger()

        self.average_power_plot = LinePlot()
        self.add_plot(self.average_power_plot, "average_power")

        self.power_histogram_plot = HBarPlot()
        self.add_plot(self.power_histogram_plot, "power_histogram")

    def pos_to_fvalue(self, pos):
        return (self.fmax - self.fmin) * (
            pos[1] - self._spectrogram_area.y
        ) / self._spectrogram_area.height + self.fmin

    def pos_to_pvalue(self, pos):
        return (self.pmax - self.pmin) * (
            pos[1] - self._power_histogram_area.y
        ) / self._power_histogram_area.height + self.pmin

    def fvalue_to_y(self, value):
        return self._spectrogram_area.y + self._spectrogram_area.height * (
            value - self.fmin
        ) / (self.fmax - self.fmin)

    def pvalue_to_y(self, value):
        return self._power_histogram_area.y + self._power_histogram_area.height * (
            value - self.pmin
        ) / (self.pmax - self.pmin)

    def update_frange_pos(self, pos1, pos2):
        v1, v2 = self.pos_to_fvalue(pos1), self.pos_to_fvalue(pos2)
        self.valid_freq_range = (
            max(self.fmin, min(v1, v2)),
            min(self.fmax, max(v1, v2)),
        )

    def update_prange_pos(self, pos1, pos2):
        v1, v2 = self.pos_to_pvalue(pos1), self.pos_to_pvalue(pos2)
        self.valid_power_range = (
            max(self.pmin, min(v1, v2)),
            min(self.pmax, max(v1, v2)),
        )
        # print(self.valid_power_range)

    def on_touch_move(self, touch):
        if self.drag_freq:
            self.update_frange_pos(self.drag_origin, touch.pos)

        elif self.drag_power:
            self.update_prange_pos(self.drag_origin, touch.pos)

        return super().on_touch_move(touch)

    def on_touch_down(self, touch):
        if self._spectrogram_area.collide_point(*touch.pos):
            self.drag_origin = touch.pos
            self.drag_freq = True
            self.update_frange_pos(self.drag_origin, touch.pos)

        elif self._power_histogram_area.collide_point(
            *touch.pos
        ) or self._power_colorbar_area.collide_point(*touch.pos):
            self.drag_origin = touch.pos
            self.drag_power = True
            self.update_prange_pos(self.drag_origin, touch.pos)

        else:
            pass  # do nothing

        return super().on_touch_down(touch)

    def on_touch_up(self, touch):
        if self.drag_freq:
            self.drag_freq = False

        elif self.drag_power:
            self.drag_power = False

        return super().on_touch_up(touch)

    def add_widget(self, widget):
        if widget in (self._plot_area, self._power_histogram_area):
            canvas = self.canvas
            self.canvas = self._fbo
        super().add_widget(widget)
        if widget in (self._plot_area, self._power_histogram_area):
            self.canvas = canvas

    def remove_widget(self, widget):
        if widget in (self._plot_area, self._power_histogram_area):
            canvas = self.canvas
            self.canvas = self._fbo
        super().remove_widget(widget)
        if widget in (self._plot_area, self._power_histogram_area):
            self.canvas = canvas

    def _get_ticks(self, major, minor, log, s_min, s_max):
        if major and s_max > s_min:
            if log:
                s_min = log10(s_min)
                s_max = log10(s_max)
                # count the decades in min - max. This is in actual decades,
                # not logs.
                n_decades = floor(s_max - s_min)
                # for the fractional part of the last decade, we need to
                # convert the log value, x, to 10**x but need to handle
                # differently if the last incomplete decade has a decade
                # boundary in it
                if floor(s_min + n_decades) != floor(s_max):
                    n_decades += 1 - (
                        10 ** (s_min + n_decades + 1) - 10 ** s_max
                    ) / 10 ** floor(s_max + 1)
                else:
                    n_decades += (
                        10 ** s_max - 10 ** (s_min + n_decades)
                    ) / 10 ** floor(s_max + 1)
                # this might be larger than what is needed, but we delete
                # excess later
                n_ticks_major = n_decades / float(major)
                n_ticks = (
                    int(floor(n_ticks_major * (minor if minor >= 1.0 else 1.0))) + 2
                )
                # in decade multiples, e.g. 0.1 of the decade, the distance
                # between ticks
                decade_dist = major / float(minor if minor else 1.0)

                points_minor = [0] * n_ticks
                points_major = [0] * n_ticks
                k = 0  # position in points major
                k2 = 0  # position in points minor
                # because each decade is missing 0.1 of the decade, if a tick
                # falls in < min_pos skip it
                min_pos = 0.1 - 0.00001 * decade_dist
                s_min_low = floor(s_min)
                # first real tick location. value is in fractions of decades
                # from the start we have to use decimals here, otherwise
                # floating point inaccuracies results in bad values
                start_dec = (
                    ceil((10 ** Decimal(s_min - s_min_low - 1)) / Decimal(decade_dist))
                    * decade_dist
                )
                count_min = 0 if not minor else floor(start_dec / decade_dist) % minor
                start_dec += s_min_low
                count = 0  # number of ticks we currently have passed start
                while True:
                    # this is the current position in decade that we are.
                    # e.g. -0.9 means that we're at 0.1 of the 10**ceil(-0.9)
                    # decade
                    pos_dec = start_dec + decade_dist * count
                    pos_dec_low = floor(pos_dec)
                    diff = pos_dec - pos_dec_low
                    zero = abs(diff) < 0.001 * decade_dist
                    if zero:
                        # the same value as pos_dec but in log scale
                        pos_log = pos_dec_low
                    else:
                        pos_log = log10((pos_dec - pos_dec_low) * 10 ** ceil(pos_dec))
                    if pos_log > s_max:
                        break
                    count += 1
                    if zero or diff >= min_pos:
                        if minor and not count_min % minor:
                            points_major[k] = pos_log
                            k += 1
                        else:
                            points_minor[k2] = pos_log
                            k2 += 1
                    count_min += 1
            else:
                # distance between each tick
                tick_dist = major / float(minor if minor else 1.0)

                rtd = log10(tick_dist)
                itd = floor(rtd)
                rtd -= itd

                if rtd < log10(2):
                    tick_dist = 2 * (10 ** itd)
                elif rtd < log10(5):
                    tick_dist = 5 * (10 ** itd)
                else:
                    tick_dist = 10 * (10 ** itd)

                r_min = ceil(s_min / tick_dist) * tick_dist
                r_max = floor(s_max / tick_dist) * tick_dist

                n_ticks = int(floor((r_max - r_min) / tick_dist) + 1)
                points_major = [0] * int(floor((r_max - r_min) / float(major)) + 1)
                points_minor = [0] * (n_ticks - len(points_major) + 1)
                k = 0  # position in points major
                k2 = 0  # position in points minor
                for m in range(0, n_ticks):
                    if minor and m % minor:
                        points_minor[k2] = m * tick_dist + r_min
                        k2 += 1
                    else:
                        points_major[k] = m * tick_dist + r_min
                        k += 1
            del points_major[k:]
            del points_minor[k2:]
        else:
            points_major = []
            points_minor = []
        return points_major, points_minor

    def _update_labels(self):
        xlabel = self._xlabel
        ylabel = self._ylabel
        flabel = self._flabel
        plabel = self._plabel
        #        x = self.x
        #        y = self.y
        #        width = self.width
        #        height = self.height
        padding = self.padding

        #        x_next = padding + x
        #        y_next = padding + y
        #        xextent = width + x
        #        yextent = height + y
        # print('{}'.format([x, y, width, height, x_next, y_next, xextent, yextent]))
        ymin = self.ymin
        ymax = self.ymax
        xmin = self.xmin
        xmax = self.xmax
        fmin = self.fmin
        fmax = self.fmax
        pmin = self.pmin
        pmax = self.pmax
        precision = self.precision
        #        x_overlap = False
        #        y_overlap = False

        x0 = self.x + self.padding
        x1 = (
            self.x
            + (self.width - self.padding * 4) * (1.0 - self.power_histogram_width)
            + self.padding
        )
        x2 = x1 + self.padding
        x3 = (
            self.x
            + (self.width - self.padding * 4) * (1.0 - self.power_colorbar_width)
            + self.padding * 2
        )
        x4 = x3 + self.padding
        x5 = self.x + self.width - self.padding
        y0 = self.y + self.padding
        y1 = self.y + (self.height - self.padding * 3) * self.average_power_height
        y2 = y1 + self.padding
        y3 = self.y + self.height - self.padding

        bounds_average_power = [x0, y0, x1, y1]
        bounds_power_histogram = [x2, y2, x3, y3]
        bounds_power_colorbar = [x4, y2, x5, y3]
        bounds_spectrogram = [x0, y2, x1, y3]
        xsz = ysz = fsz = psz = (0, 0)  # 回転前のサイズ

        # set up x and y axis labels
        if xlabel:
            xlabel.text = self.xlabel
            xlabel.texture_update()
            xsz = xlabel.size = xlabel.texture_size
        if ylabel:
            ylabel.text = self.ylabel
            ylabel.texture_update()
            ysz = ylabel.size = ylabel.texture_size
        if flabel:
            flabel.text = self.flabel
            flabel.texture_update()
            fsz = flabel.size = flabel.texture_size
        if plabel:
            plabel.text = self.plabel
            plabel.texture_update()
            psz = plabel.size = plabel.texture_size

        if xlabel:
            xlabel.x = x0 + (x1 - x0 + max(ysz[1], fsz[1])) / 2 - xsz[0] / 2
            xlabel.y = y0
        if ylabel:
            ylabel.x = x0 + max(ysz[1], fsz[1]) / 2 - ysz[0] / 2
            ylabel.y = y0 + (y1 - y0 + xsz[1] - ysz[1]) / 2
            ylabel.angle = 45
        if flabel:
            flabel.x = x0 + max(ysz[1], fsz[1]) / 2 - fsz[0] / 2
            flabel.y = y2 + (y3 - y2 - fsz[1]) / 2
            flabel.angle = 45
        if plabel:
            plabel.x = x5 - (psz[0] + psz[1]) / 2
            plabel.y = y2 + (y3 - y2 - psz[1]) / 2
            plabel.angle = -45

        bounds_average_power[0] += max(ysz[1], fsz[1]) + self.padding
        bounds_average_power[1] += xsz[1] + self.padding
        bounds_spectrogram[0] += max(ysz[1], fsz[1]) + self.padding
        bounds_power_histogram[2] -= psz[1] + self.padding
        bounds_power_colorbar[0] -= psz[1] + self.padding
        bounds_power_colorbar[2] -= psz[1] + self.padding

        xpoints = self._ticks_majorx
        xlabels = self._x_grid_label
        xlabel_grid = self.x_grid_label
        ypoints = self._ticks_majory
        ylabels = self._y_grid_label
        ylabel_grid = self.y_grid_label
        fpoints = self._ticks_majorf
        flabels = self._f_grid_label
        flabel_grid = self.f_grid_label
        ppoints = self._ticks_majorp
        plabels = self._p_grid_label
        plabel_grid = self.p_grid_label

        # modify by value labels' size
        xlsz = (0, 0)
        ylsz = (0, 0)
        flsz = (0, 0)
        plsz = (0, 0)
        if len(xlabels) and xlabel_grid:
            funcexp = exp10 if self.xlog else identity
            funclog = log10 if self.xlog else identity
            # find the distance from the end that'll fit the last tick label
            xlabels[0].text = precision % funcexp(xpoints[-1])
            xlabels[0].texture_update()
            xlsz = xlabels[0].texture_size
        if len(ylabels) and ylabel_grid:
            funcexp = exp10 if self.ylog else identity
            funclog = log10 if self.ylog else identity
            # find the distance from the end that'll fit the last tick label
            ylabels[0].text = precision % funcexp(ypoints[-1])
            ylabels[0].texture_update()
            ylsz = ylabels[0].texture_size
        if len(flabels) and flabel_grid:
            funcexp = exp10 if self.flog else identity
            funclog = log10 if self.flog else identity
            # find the distance from the end that'll fit the last tick label
            flabels[0].text = precision % funcexp(fpoints[-1])
            flabels[0].texture_update()
            flsz = ylabels[0].texture_size
        if len(plabels) and plabel_grid:
            funcexp = exp10 if self.plog else identity
            funclog = log10 if self.plog else identity
            # find the distance from the end that'll fit the last tick label
            plabels[0].text = precision % funcexp(ppoints[-1])
            plabels[0].texture_update()
            plsz = ylabels[0].texture_size

        bounds_average_power[0] += max(ylsz[0], flsz[0]) + self.padding
        bounds_average_power[1] += xlsz[1] + self.padding
        bounds_spectrogram[0] += max(ylsz[0], flsz[0]) + self.padding
        bounds_power_histogram[2] -= plsz[0] + self.padding
        bounds_power_colorbar[0] -= plsz[0] + self.padding
        bounds_power_colorbar[2] -= plsz[0] + self.padding

        # now x and y tick mark labels
        if len(ylabels) and ylabel_grid:
            # horizontal size of the largest tick label, to have enough room
            funcexp = exp10 if self.ylog else identity
            funclog = log10 if self.ylog else identity
            #            ylabels[0].text = precision % funcexp(ypoints[0])
            #            ylabels[0].texture_update()
            #            y1 = ylabels[0].texture_size
            #            y_start = y_next + (padding + y1[1] if len(xlabels) and xlabel_grid
            #                                else 0) + \
            #                               (padding + y1[1] if not y_next else 0)
            #            yextent = y + height * self.average_power_height - padding - y1[1] / 2.

            ymin = funclog(ymin)
            #            ratio = (yextent - y_start) / float(funclog(ymax) - ymin)
            ratio = (bounds_average_power[3] - bounds_average_power[1]) / float(
                funclog(ymax) - ymin
            )
            #            y_start -= y1[1] / 2.
            #            y1 = y1[0]
            for k in range(len(ylabels)):
                ylabels[k].text = precision % funcexp(ypoints[k])
                ylabels[k].texture_update()
                ylabels[k].size = ylabels[k].texture_size
                #                y1 = max(y1, ylabels[k].texture_size[0])
                #                ylabels[k].pos = (
                #                    int(x_next),
                #                    int(y_start + (ypoints[k] - ymin) * ratio))
                ylabels[k].pos = (
                    bounds_average_power[0] - ylsz[0] - self.padding,
                    int(
                        bounds_average_power[1]
                        + (ypoints[k] - ymin) * ratio
                        - ylsz[1] / 2.0
                    ),
                )
        #            if len(ylabels) > 1 and ylabels[0].top > ylabels[1].y:
        #                y_overlap = True
        #            else:
        #                x_next += y1 + padding
        if len(flabels) and flabel_grid:
            # horizontal size of the largest tick label, to have enough room
            funcexp = exp10 if self.flog else identity
            funclog = log10 if self.flog else identity
            fmin = funclog(fmin)
            ratio = (bounds_spectrogram[3] - bounds_spectrogram[1]) / float(
                funclog(fmax) - fmin
            )
            for k in range(1, len(flabels)):
                flabels[k].text = precision % funcexp(fpoints[k])
                flabels[k].texture_update()
                flabels[k].size = flabels[k].texture_size
                flabels[k].pos = (
                    bounds_spectrogram[0] - flsz[0] - self.padding,
                    int(
                        bounds_spectrogram[1]
                        + (fpoints[k] - fmin) * ratio
                        - flsz[1] / 2.0
                    ),
                )
            flabels[0].text = ""
            flabels[0].pos = (0, 0)
            flabels[0].size = (0, 0)

        if len(plabels) and plabel_grid:
            # horizontal size of the largest tick label, to have enough room
            funcexp = exp10 if self.plog else identity
            funclog = log10 if self.plog else identity
            pmin = funclog(pmin)
            ratio = (bounds_spectrogram[3] - bounds_spectrogram[1]) / float(
                funclog(pmax) - pmin
            )
            for k in range(0, len(plabels)):
                plabels[k].text = precision % funcexp(ppoints[k])
                plabels[k].texture_update()
                plabels[k].size = plabels[k].texture_size
                plabels[k].pos = (
                    bounds_power_colorbar[2] + self.padding,
                    int(
                        bounds_power_colorbar[1]
                        + (ppoints[k] - pmin) * ratio
                        - plsz[1] / 2.0
                    ),
                )

        if len(xlabels) and xlabel_grid:
            funcexp = exp10 if self.xlog else identity
            funclog = log10 if self.xlog else identity
            # find the distance from the end that'll fit the last tick label
            #            xlabels[0].text = precision % funcexp(xpoints[-1])
            #            xlabels[0].texture_update()
            #            xextent = x + width - xlabels[0].texture_size[0] / 2. - padding
            # find the distance from the start that'll fit the first tick label
            #            if not x_next:
            #                xlabels[0].text = precision % funcexp(xpoints[0])
            #                xlabels[0].texture_update()
            #                x_next = padding + xlabels[0].texture_size[0] / 2.
            xmin = funclog(xmin)
            #            ratio = (xextent - x_next) / float(funclog(self.xmax) - xmin)
            ratio = (bounds_average_power[2] - bounds_average_power[0]) / float(
                funclog(self.xmax) - xmin
            )
            #            right = -1
            for k in range(len(xlabels)):
                xlabels[k].text = precision % funcexp(xpoints[k])
                # update the size so we can center the labels on ticks
                xlabels[k].texture_update()
                xlabels[k].size = xlabels[k].texture_size
                half_ts = xlabels[k].texture_size[0] / 2.0
                xlabels[k].pos = (
                    bounds_average_power[0]
                    + int((xpoints[k] - xmin) * ratio - half_ts),
                    bounds_average_power[1] - xlsz[1] - self.padding,
                )
        #                if xlabels[k].x < right:
        #                    x_overlap = True
        #                    break
        #                right = xlabels[k].right
        #            if not x_overlap:
        #                y_next += padding + xlabels[0].texture_size[1]
        # now re-center the x and y axis labels
        #        if x_overlap:
        #            for k in range(len(xlabels)):
        #                xlabels[k].text = ''
        #        if y_overlap:
        #            for k in range(len(ylabels)):
        #                ylabels[k].text = ''
        return (
            bounds_average_power,
            bounds_spectrogram,
            bounds_power_histogram,
            bounds_power_colorbar,
        )

    def _update_ticks(self, size, ssize, hsize):
        # re-compute the positions of the bounding rectangle
        if self.valid_freq_range[0] is not None:
            vfsize = (
                ssize[0],
                self.fvalue_to_y(self.valid_freq_range[0]),
                ssize[2],
                self.fvalue_to_y(self.valid_freq_range[1]),
            )
        else:
            vfsize = (0, 0, 0, 0)

        if self.valid_power_range[0] is not None:
            vpsize = (
                hsize[0],
                self.pvalue_to_y(self.valid_power_range[0]),
                hsize[2],
                self.pvalue_to_y(self.valid_power_range[1]),
            )
        else:
            vpsize = (0, 0, 0, 0)

        for mesh, sz in zip(
            (
                self._waveform_mesh_rect,
                self._spectrogram_mesh_rect,
                self._histogram_mesh_rect,
                self._valid_freq_area_mesh,
                self._valid_power_area_mesh,
            ),
            (size, ssize, hsize, vfsize, vpsize),
        ):
            vert = mesh.vertices
            if self.draw_border:
                s0, s1, s2, s3 = sz
                vert = mesh.vertices
                vert[0] = s0
                vert[1] = s1
                vert[4] = s2
                vert[5] = s1
                vert[8] = s2
                vert[9] = s3
                vert[12] = s0
                vert[13] = s3
                vert[16] = s0
                vert[17] = s1
            else:
                vert[0:18] = [0 for k in range(18)]
            mesh.vertices = vert
        # re-compute the positions of the x/y axis ticks
        mesh = self._mesh_ticks
        vert = mesh.vertices
        start = 0
        xpoints = self._ticks_majorx
        ypoints = self._ticks_majory
        xpoints2 = self._ticks_minorx
        ypoints2 = self._ticks_minory
        ylog = self.ylog
        xlog = self.xlog
        xmin = self.xmin
        xmax = self.xmax
        if xlog:
            xmin = log10(xmin)
            xmax = log10(xmax)
        ymin = self.ymin
        ymax = self.ymax
        if ylog:
            ymin = log10(ymin)
            ymax = log10(ymax)
        if len(xpoints):
            top = size[3] if self.x_grid else metrics.dp(12) + size[1]
            ratio = (size[2] - size[0]) / float(xmax - xmin)
            for k in range(start, len(xpoints) + start):
                vert[k * 8] = size[0] + (xpoints[k - start] - xmin) * ratio
                vert[k * 8 + 1] = size[1]
                vert[k * 8 + 4] = vert[k * 8]
                vert[k * 8 + 5] = top
            start += len(xpoints)
        if len(xpoints2):
            top = metrics.dp(8) + size[1]
            ratio = (size[2] - size[0]) / float(xmax - xmin)
            for k in range(start, len(xpoints2) + start):
                vert[k * 8] = size[0] + (xpoints2[k - start] - xmin) * ratio
                vert[k * 8 + 1] = size[1]
                vert[k * 8 + 4] = vert[k * 8]
                vert[k * 8 + 5] = top
            start += len(xpoints2)
        if len(ypoints):
            top = size[2] if self.y_grid else metrics.dp(12) + size[0]
            ratio = (size[3] - size[1]) / float(ymax - ymin)
            for k in range(start, len(ypoints) + start):
                vert[k * 8 + 1] = size[1] + (ypoints[k - start] - ymin) * ratio
                vert[k * 8 + 5] = vert[k * 8 + 1]
                vert[k * 8] = size[0]
                vert[k * 8 + 4] = top
            start += len(ypoints)
        if len(ypoints2):
            top = metrics.dp(8) + size[0]
            ratio = (size[3] - size[1]) / float(ymax - ymin)
            for k in range(start, len(ypoints2) + start):
                vert[k * 8 + 1] = size[1] + (ypoints2[k - start] - ymin) * ratio
                vert[k * 8 + 5] = vert[k * 8 + 1]
                vert[k * 8] = size[0]
                vert[k * 8 + 4] = top
        mesh.vertices = vert

    x_axis = ListProperty([None])
    y_axis = ListProperty([None])
    p_axis = ListProperty([None])
    h_axis = ListProperty([None])

    def get_x_axis(self, axis=0):
        if axis == 0:
            return self.xlog, self.xmin, self.xmax
        info = self.x_axis[axis]
        return (info["log"], info["min"], info["max"])

    def get_y_axis(self, axis=0):
        if axis == 0:
            return self.ylog, self.ymin, self.ymax
        info = self.y_axis[axis]
        return (info["log"], info["min"], info["max"])

    def get_p_axis(self, axis=0):
        if axis == 0:
            return self.plog, self.pmin, self.pmax
        info = self.p_axis[axis]
        return (info["log"], info["min"], info["max"])

    def get_h_axis(self, axis=0):
        if axis == 0:
            return self.hlog, self.hmin, self.hmax
        info = self.h_axis[axis]
        return (info["log"], info["min"], info["max"])

    def add_x_axis(self, xmin, xmax, xlog=False):
        data = {"log": xlog, "min": xmin, "max": xmax}
        self.x_axis.append(data)
        return data

    def add_y_axis(self, ymin, ymax, ylog=False):
        data = {"log": ylog, "min": ymin, "max": ymax}
        self.y_axis.append(data)
        return data

    def _update_plot(self, size, target):
        if target == "average_power":
            plot = self.average_power_plot
            xlog, xmin, xmax = self.get_x_axis(plot.x_axis)
            ylog, ymin, ymax = self.get_y_axis(plot.y_axis)
            plot._update(xlog, xmin, xmax, ylog, ymin, ymax, size)
        elif target == "power_histogram":
            plot = self.power_histogram_plot
            xlog, xmin, xmax = self.get_h_axis(plot.x_axis)
            ylog, ymin, ymax = self.get_p_axis(plot.y_axis)
            plot._update(xlog, xmax, xmin, ylog, ymin, ymax, size)

    def _update_colors(self, *args):
        self._mesh_ticks_color.rgba = tuple(self.tick_color)
        self._background_color.rgba = tuple(self.background_color)
        self._mesh_rect_color.rgba = tuple(self.border_color)
        self._valid_power_area_color.rgba = tuple(self.valid_area_color)
        self._valid_freq_area_color.rgba = tuple(self.valid_area_color)

    def update_colorbar_drawings(self, *args):
        if self._power_colorbar_texture is None:
            self._power_colorbar_texture = Texture.create(size=(1, 2), colorfmt="rgba")
            self._power_colorbar_texture.blit_buffer(
                bytes([0, 0, 0, 255, 255, 255, 255, 255]),
                colorfmt="rgba",
                bufferfmt="ubyte",
            )

        w = self._power_colorbar_area
        width = w.size[0]
        height = w.size[1]

        if (
            self.valid_power_range[0] is not None
            and self.valid_power_range[1] > self.valid_power_range[0]
        ):
            min = (
                w.size[1]
                * (self.valid_power_range[0] - self.pmin)
                / (self.pmax - self.pmin)
            )
            max = (
                w.size[1]
                * (self.valid_power_range[1] - self.pmin)
                / (self.pmax - self.pmin)
            )
        else:
            min = 0
            max = w.size[1]

        w.canvas.clear()
        with w.canvas:
            Rectangle(
                pos=(w.pos[0], w.pos[1] + min),
                size=(width, max - min),
                texture=self._power_colorbar_texture,
            )
            Color(0.0, 0.0, 0.0, 1.0)
            Rectangle(pos=(w.pos[0], w.pos[1]), size=(width, min))
            Color(1.0, 1.0, 1.0, 1.0)
            Rectangle(pos=(w.pos[0], w.pos[1] + max), size=(width, height - max))

    def _redraw_all(self, *args):
        # add/remove all the required labels
        xpoints_major, xpoints_minor = self._redraw_x(*args)
        ypoints_major, ypoints_minor = self._redraw_y(*args)
        fpoints_major, fpoints_minor = self._redraw_f(*args)
        ppoints_major, ppoints_minor = self._redraw_p(*args)

        mesh = self._mesh_ticks
        n_points = (
            len(xpoints_major)
            + len(xpoints_minor)
            + len(ypoints_major)
            + len(ypoints_minor)
        )
        mesh.vertices = [0] * (n_points * 8)
        mesh.indices = [k for k in range(n_points * 2)]
        self._redraw_size()

    def _redraw_x(self, *args):
        font_size = self.font_size
        if self.xlabel:
            xlabel = self._xlabel
            if not xlabel:
                xlabel = Label()
                self.add_widget(xlabel)
                self._xlabel = xlabel

            xlabel.font_size = font_size
            for k, v in self.label_options.items():
                setattr(xlabel, k, v)

        else:
            xlabel = self._xlabel
            if xlabel:
                self.remove_widget(xlabel)
                self._xlabel = None
        grids = self._x_grid_label
        xpoints_major, xpoints_minor = self._get_ticks(
            self.x_ticks_major, self.x_ticks_minor, self.xlog, self.xmin, self.xmax
        )
        self._ticks_majorx = xpoints_major
        self._ticks_minorx = xpoints_minor
        # print('{}, {}'.format(self._ticks_majorx, self._ticks_minorx))

        if not self.x_grid_label:
            n_labels = 0
        else:
            n_labels = len(xpoints_major)

        for k in range(n_labels, len(grids)):
            self.remove_widget(grids[k])
        del grids[n_labels:]

        grid_len = len(grids)
        grids.extend([None] * (n_labels - len(grids)))
        for k in range(grid_len, n_labels):
            grids[k] = GraphRotatedLabel(
                font_size=font_size, angle=self.x_ticks_angle, **self.label_options
            )
            self.add_widget(grids[k])
        return xpoints_major, xpoints_minor

    def _redraw_y(self, *args):
        font_size = self.font_size
        if self.ylabel:
            ylabel = self._ylabel
            if not ylabel:
                ylabel = GraphRotatedLabel()
                self.add_widget(ylabel)
                self._ylabel = ylabel

            ylabel.font_size = font_size
            for k, v in self.label_options.items():
                setattr(ylabel, k, v)
        else:
            ylabel = self._ylabel
            if ylabel:
                self.remove_widget(ylabel)
                self._ylabel = None
        grids = self._y_grid_label
        ypoints_major, ypoints_minor = self._get_ticks(
            self.y_ticks_major, self.y_ticks_minor, self.ylog, self.ymin, self.ymax
        )
        self._ticks_majory = ypoints_major
        self._ticks_minory = ypoints_minor

        if not self.y_grid_label:
            n_labels = 0
        else:
            n_labels = len(ypoints_major)

        for k in range(n_labels, len(grids)):
            self.remove_widget(grids[k])
        del grids[n_labels:]

        grid_len = len(grids)
        grids.extend([None] * (n_labels - len(grids)))
        for k in range(grid_len, n_labels):
            grids[k] = Label(font_size=font_size, **self.label_options)
            self.add_widget(grids[k])
        return ypoints_major, ypoints_minor

    def _redraw_f(self, *args):
        font_size = self.font_size
        if self.flabel:
            flabel = self._flabel
            if not flabel:
                flabel = GraphRotatedLabel()
                self.add_widget(flabel)
                self._flabel = flabel

            flabel.font_size = font_size
            for k, v in self.label_options.items():
                setattr(flabel, k, v)
        else:
            flabel = self._flabel
            if flabel:
                self.remove_widget(flabel)
                self._flabel = None
        grids = self._f_grid_label
        fpoints_major, fpoints_minor = self._get_ticks(
            self.f_ticks_major, self.f_ticks_minor, self.flog, self.fmin, self.fmax
        )
        self._ticks_majorf = fpoints_major
        self._ticks_minorf = fpoints_minor
        # print('{}, {}, {}, {}, {}, {}, {}'.format(self.f_ticks_major, self.f_ticks_minor, self.flog, self.fmin, self.fmax, self._ticks_majorf, self._ticks_minorf))

        if not self.f_grid_label:
            n_labels = 0
        else:
            n_labels = len(fpoints_major)

        for k in range(n_labels, len(grids)):
            self.remove_widget(grids[k])
        del grids[n_labels:]

        grid_len = len(grids)
        grids.extend([None] * (n_labels - len(grids)))
        for k in range(grid_len, n_labels):
            grids[k] = Label(font_size=font_size, **self.label_options)
            self.add_widget(grids[k])
        return fpoints_major, fpoints_minor

    def _redraw_p(self, *args):
        font_size = self.font_size
        if self.plabel:
            plabel = self._plabel
            if not plabel:
                plabel = GraphRotatedLabel()
                self.add_widget(plabel)
                self._plabel = plabel

            plabel.font_size = font_size
            for k, v in self.label_options.items():
                setattr(plabel, k, v)
        else:
            plabel = self._plabel
            if plabel:
                self.remove_widget(plabel)
                self._plabel = None
        grids = self._p_grid_label
        ppoints_major, ppoints_minor = self._get_ticks(
            self.p_ticks_major, self.p_ticks_minor, self.plog, self.pmin, self.pmax
        )
        self._ticks_majorp = ppoints_major
        self._ticks_minorp = ppoints_minor

        if not self.p_grid_label:
            n_labels = 0
        else:
            n_labels = len(ppoints_major)

        for k in range(n_labels, len(grids)):
            self.remove_widget(grids[k])
        del grids[n_labels:]

        grid_len = len(grids)
        grids.extend([None] * (n_labels - len(grids)))
        for k in range(grid_len, n_labels):
            grids[k] = Label(font_size=font_size, **self.label_options)
            self.add_widget(grids[k])
        return ppoints_major, ppoints_minor

    def _redraw_size(self, *args):
        # size a 4-tuple describing the bounding box in which we can draw
        # graphs, it's (x0, y0, x1, y1), which correspond with the bottom left
        # and top right corner locations, respectively
        self._clear_buffer()
        size, ssize, hsize, csize = self._update_labels()
        self._plot_area.pos = (size[0], size[1])
        self._plot_area.size = (size[2] - size[0], size[3] - size[1])
        self._spectrogram_area.pos = (ssize[0], ssize[1])
        self._spectrogram_area.size = (ssize[2] - ssize[0] - 1, ssize[3] - ssize[1] - 1)
        self._power_histogram_area.pos = (hsize[0], hsize[1])
        self._power_histogram_area.size = (
            hsize[2] - hsize[0] - 1,
            hsize[3] - hsize[1] - 1,
        )
        self._power_colorbar_area.pos = (csize[0], csize[1])
        self._power_colorbar_area.size = (
            csize[2] - csize[0] - 1,
            csize[3] - csize[1] - 1,
        )
        # print(csize)

        if self.size[0] and self.size[1]:
            self._fbo.size = self.size
        else:
            self._fbo.size = 1, 1  # gl errors otherwise
        self._fbo_rect.texture = self._fbo.texture
        self._fbo_rect.size = self.size
        self._fbo_rect.pos = self.pos
        self._background_rect.size = self.size
        self._update_ticks(size, ssize, hsize)
        self._update_plot(size, "average_power")
        self._update_plot(hsize, "power_histogram")
        self.update_colorbar_drawings()

    def _clear_buffer(self, *largs):
        fbo = self._fbo
        fbo.bind()
        fbo.clear_buffer()
        fbo.release()

    def add_plot(self, plot, target):
        """Add a new plot to this graph.

        :Parameters:
            `plot`:
                Plot to add to this graph.

        >>> graph = Graph()
        >>> plot = MeshLinePlot(mode='line_strip', color=[1, 0, 0, 1])
        >>> plot.points = [(x / 10., sin(x / 50.)) for x in range(-0, 101)]
        >>> graph.add_plot(plot)
        """
        if plot in self.plots:
            return

        if target == "average_power":
            add = self._plot_area.canvas.add
        elif target == "power_histogram":
            add = self._power_histogram_area.canvas.add
        else:
            add = None

        for instr in plot.get_drawings():
            add(instr)
        plot.bind(on_clear_plot=self._clear_buffer)
        self.plots.append(plot)

    def remove_plot(self, plot, target):
        """Remove a plot from this graph.

        :Parameters:
            `plot`:
                Plot to remove from this graph.

        >>> graph = Graph()
        >>> plot = MeshLinePlot(mode='line_strip', color=[1, 0, 0, 1])
        >>> plot.points = [(x / 10., sin(x / 50.)) for x in range(-0, 101)]
        >>> graph.add_plot(plot)
        >>> graph.remove_plot(plot)
        """
        if plot not in self.plots:
            return
        if target == "average_power":
            remove = self._plot_area.canvas.remove
        elif target == "power_histogram":
            remove = self._power_histogram_area.canvas.remove
        else:
            remove = None

        for instr in plot.get_drawings():
            remove(instr)
        plot.unbind(on_clear_plot=self._clear_buffer)
        self.plots.remove(plot)
        self._clear_buffer()

    def collide_plot(self, x, y):
        """Determine if the given coordinates fall inside the plot area. Use
        `x, y = self.to_widget(x, y, relative=True)` to first convert into
        widget coordinates if it's in window coordinates because it's assumed
        to be given in local widget coordinates, relative to the graph's pos.

        :Parameters:
            `x, y`:
                The coordinates to test.
        """
        adj_x, adj_y = x - self._plot_area.pos[0], y - self._plot_area.pos[1]
        return (
            0 <= adj_x <= self._plot_area.size[0]
            and 0 <= adj_y <= self._plot_area.size[1]
        )

    def to_data(self, x, y):
        """Convert widget coords to data coords. Use
        `x, y = self.to_widget(x, y, relative=True)` to first convert into
        widget coordinates if it's in window coordinates because it's assumed
        to be given in local widget coordinates, relative to the graph's pos.

        :Parameters:
            `x, y`:
                The coordinates to convert.

        If the graph has multiple axes, use :class:`Plot.unproject` instead.
        """
        adj_x = float(x - self._plot_area.pos[0])
        adj_y = float(y - self._plot_area.pos[1])
        norm_x = adj_x / self._plot_area.size[0]
        norm_y = adj_y / self._plot_area.size[1]
        if self.xlog:
            xmin, xmax = log10(self.xmin), log10(self.xmax)
            conv_x = 10.0 ** (norm_x * (xmax - xmin) + xmin)
        else:
            conv_x = norm_x * (self.xmax - self.xmin) + self.xmin
        if self.ylog:
            ymin, ymax = log10(self.ymin), log10(self.ymax)
            conv_y = 10.0 ** (norm_y * (ymax - ymin) + ymin)
        else:
            conv_y = norm_y * (self.ymax - self.ymin) + self.ymin
        return [conv_x, conv_y]

    xmin = NumericProperty(0.0)
    """The x-axis minimum value.

    If :data:`xlog` is True, xmin must be larger than zero.

    :data:`xmin` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    xmax = NumericProperty(100.0)
    """The x-axis maximum value, larger than xmin.

    :data:`xmax` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    xlog = BooleanProperty(False)
    """Determines whether the x-axis should be displayed logarithmically (True)
    or linearly (False).

    :data:`xlog` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    x_ticks_major = BoundedNumericProperty(64, min=0)
    """Distance between major tick marks on the x-axis.

    Determines the distance between the major tick marks. Major tick marks
    start from min and re-occur at every ticks_major until :data:`xmax`.
    If :data:`xmax` doesn't overlap with a integer multiple of ticks_major,
    no tick will occur at :data:`xmax`. Zero indicates no tick marks.

    If :data:`xlog` is true, then this indicates the distance between ticks
    in multiples of current decade. E.g. if :data:`xmin` is 0.1 and
    ticks_major is 0.1, it means there will be a tick at every 10th of the
    decade, i.e. 0.1 ... 0.9, 1, 2... If it is 0.3, the ticks will occur at
    0.1, 0.3, 0.6, 0.9, 2, 5, 8, 10. You'll notice that it went from 8 to 10
    instead of to 20, that's so that we can say 0.5 and have ticks at every
    half decade, e.g. 0.1, 0.5, 1, 5, 10, 50... Similarly, if ticks_major is
    1.5, there will be ticks at 0.1, 5, 100, 5,000... Also notice, that there's
    always a major tick at the start. Finally, if e.g. :data:`xmin` is 0.6
    and this 0.5 there will be ticks at 0.6, 1, 5...

    :data:`x_ticks_major` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    x_ticks_minor = BoundedNumericProperty(0, min=0)
    """The number of sub-intervals that divide x_ticks_major.

    Determines the number of sub-intervals into which ticks_major is divided,
    if non-zero. The actual number of minor ticks between the major ticks is
    ticks_minor - 1. Only used if ticks_major is non-zero. If there's no major
    tick at xmax then the number of minor ticks after the last major
    tick will be however many ticks fit until xmax.

    If self.xlog is true, then this indicates the number of intervals the
    distance between major ticks is divided. The result is the number of
    multiples of decades between ticks. I.e. if ticks_minor is 10, then if
    ticks_major is 1, there will be ticks at 0.1, 0.2...0.9, 1, 2, 3... If
    ticks_major is 0.3, ticks will occur at 0.1, 0.12, 0.15, 0.18... Finally,
    as is common, if ticks major is 1, and ticks minor is 5, there will be
    ticks at 0.1, 0.2, 0.4... 0.8, 1, 2...

    :data:`x_ticks_minor` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    x_grid = BooleanProperty(False)
    """Determines whether the x-axis has tick marks or a full grid.

    If :data:`x_ticks_major` is non-zero, then if x_grid is False tick marks
    will be displayed at every major tick. If x_grid is True, instead of ticks,
    a vertical line will be displayed at every major tick.

    :data:`x_grid` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    x_grid_label = BooleanProperty(True)
    """Whether labels should be displayed beneath each major tick. If true,
    each major tick will have a label containing the axis value.

    :data:`x_grid_label` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to False.
    """

    xlabel = StringProperty("Time [Frame]")
    """The label for the x-axis. If not empty it is displayed in the center of
    the axis.

    :data:`xlabel` is a :class:`~kivy.properties.StringProperty`,
    defaults to ''.
    """

    ymin = NumericProperty(0.0)
    """The y-axis minimum value.

    If :data:`ylog` is True, ymin must be larger than zero.

    :data:`ymin` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    ymax = NumericProperty(100.0)
    """The y-axis maximum value, larger than ymin.

    :data:`ymax` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    ylog = BooleanProperty(False)
    """Determines whether the y-axis should be displayed logarithmically (True)
    or linearly (False).

    :data:`ylog` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    y_ticks_major = BoundedNumericProperty(10, min=0)
    """Distance between major tick marks. See :data:`x_ticks_major`.

    :data:`y_ticks_major` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    y_ticks_minor = BoundedNumericProperty(0, min=0)
    """The number of sub-intervals that divide ticks_major.
    See :data:`x_ticks_minor`.

    :data:`y_ticks_minor` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    y_grid = BooleanProperty(False)
    """Determines whether the y-axis has tick marks or a full grid. See
    :data:`x_grid`.

    :data:`y_grid` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    y_grid_label = BooleanProperty(True)
    """Whether labels should be displayed beneath each major tick. If true,
    each major tick will have a label containing the axis value.

    :data:`y_grid_label` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to False.
    """

    ylabel = StringProperty("Average Power")
    """The label for the y-axis. If not empty it is displayed in the center of
    the axis.

    :data:`ylabel` is a :class:`~kivy.properties.StringProperty`,
    defaults to ''.
    """

    fmin = NumericProperty(0.0)
    """The f-axis minimum value.

    If :data:`flog` is True, fmin must be larger than zero.

    :data:`fmin` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    fmax = NumericProperty(100.0)
    """The f-axis maximum value, larger than fmin.

    :data:`fmax` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    flog = BooleanProperty(False)
    """Determines whether the f-axis should be displayed logarithmically (True)
    or linearly (False).

    :data:`flog` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    f_ticks_major = BoundedNumericProperty(32, min=0)
    """Distance between major tick marks on the f-axis.

    Determines the distance between the major tick marks. Major tick marks
    start from min and re-occur at every ticks_major until :data:`fmax`.
    If :data:`fmax` doesn't overlap with a integer multiple of ticks_major,
    no tick will occur at :data:`fmax`. Zero indicates no tick marks.

    If :data:`flog` is true, then this indicates the distance between ticks
    in multiples of current decade. E.g. if :data:`fmin` is 0.1 and
    ticks_major is 0.1, it means there will be a tick at every 10th of the
    decade, i.e. 0.1 ... 0.9, 1, 2... If it is 0.3, the ticks will occur at
    0.1, 0.3, 0.6, 0.9, 2, 5, 8, 10. You'll notice that it went from 8 to 10
    instead of to 20, that's so that we can say 0.5 and have ticks at every
    half decade, e.g. 0.1, 0.5, 1, 5, 10, 50... Similarly, if ticks_major is
    1.5, there will be ticks at 0.1, 5, 100, 5,000... Also notice, that there's
    always a major tick at the start. Finally, if e.g. :data:`fmin` is 0.6
    and this 0.5 there will be ticks at 0.6, 1, 5...

    :data:`f_ticks_major` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    f_ticks_minor = BoundedNumericProperty(0, min=0)
    """The number of sub-intervals that divide f_ticks_major.

    Determines the number of sub-intervals into which ticks_major is divided,
    if non-zero. The actual number of minor ticks between the major ticks is
    ticks_minor - 1. Only used if ticks_major is non-zero. If there's no major
    tick at fmax then the number of minor ticks after the last major
    tick will be however many ticks fit until fmax.

    If self.flog is true, then this indicates the number of intervals the
    distance between major ticks is divided. The result is the number of
    multiples of decades between ticks. I.e. if ticks_minor is 10, then if
    ticks_major is 1, there will be ticks at 0.1, 0.2...0.9, 1, 2, 3... If
    ticks_major is 0.3, ticks will occur at 0.1, 0.12, 0.15, 0.18... Finally,
    as is common, if ticks major is 1, and ticks minor is 5, there will be
    ticks at 0.1, 0.2, 0.4... 0.8, 1, 2...

    :data:`f_ticks_minor` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    f_grid = BooleanProperty(False)
    """Determines whether the f-axis has tick marks or a full grid.

    If :data:`f_ticks_major` is non-zero, then if f_grid is False tick marks
    will be displayed at every major tick. If f_grid is True, instead of ticks,
    a vertical line will be displayed at every major tick.

    :data:`f_grid` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    f_grid_label = BooleanProperty(True)
    """Whether labels should be displayed beneath each major tick. If true,
    each major tick will have a label containing the axis value.

    :data:`f_grid_label` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to False.
    """

    flabel = StringProperty("Frequency Bin")
    """The label for the f-axis. If not empty it is displayed in the center of
    the axis.

    :data:`flabel` is a :class:`~kivy.properties.StringProperty`,
    defaults to ''.
    """

    pmin = NumericProperty(0.0)
    """The p-axis minimum value.

    If :data:`plog` is True, pmin must be larger than zero.

    :data:`pmin` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    pmax = NumericProperty(100.0)
    """The p-axis maximum value, larger than pmin.

    :data:`pmax` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    plog = BooleanProperty(False)
    """Determines whether the p-axis should be displayed logarithmically (True)
    or linearly (False).

    :data:`plog` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    p_ticks_major = BoundedNumericProperty(8, min=0)
    """Distance between major tick marks on the p-axis.

    Determines the distance between the major tick marks. Major tick marks
    start from min and re-occur at every ticks_major until :data:`pmax`.
    If :data:`pmax` doesn't overlap with a integer multiple of ticks_major,
    no tick will occur at :data:`pmax`. Zero indicates no tick marks.

    If :data:`plog` is true, then this indicates the distance between ticks
    in multiples of current decade. E.g. if :data:`pmin` is 0.1 and
    ticks_major is 0.1, it means there will be a tick at every 10th of the
    decade, i.e. 0.1 ... 0.9, 1, 2... If it is 0.3, the ticks will occur at
    0.1, 0.3, 0.6, 0.9, 2, 5, 8, 10. You'll notice that it went from 8 to 10
    instead of to 20, that's so that we can say 0.5 and have ticks at every
    half decade, e.g. 0.1, 0.5, 1, 5, 10, 50... Similarly, if ticks_major is
    1.5, there will be ticks at 0.1, 5, 100, 5,000... Also notice, that there's
    always a major tick at the start. Finally, if e.g. :data:`pmin` is 0.6
    and this 0.5 there will be ticks at 0.6, 1, 5...

    :data:`p_ticks_major` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    p_ticks_minor = BoundedNumericProperty(0, min=0)
    """The number of sub-intervals that divide p_ticks_major.

    Determines the number of sub-intervals into which ticks_major is divided,
    if non-zero. The actual number of minor ticks between the major ticks is
    ticks_minor - 1. Only used if ticks_major is non-zero. If there's no major
    tick at pmax then the number of minor ticks after the last major
    tick will be however many ticks fit until pmax.

    If self.plog is true, then this indicates the number of intervals the
    distance between major ticks is divided. The result is the number of
    multiples of decades between ticks. I.e. if ticks_minor is 10, then if
    ticks_major is 1, there will be ticks at 0.1, 0.2...0.9, 1, 2, 3... If
    ticks_major is 0.3, ticks will occur at 0.1, 0.12, 0.15, 0.18... Finally,
    as is common, if ticks major is 1, and ticks minor is 5, there will be
    ticks at 0.1, 0.2, 0.4... 0.8, 1, 2...

    :data:`p_ticks_minor` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    p_grid = BooleanProperty(False)
    """Determines whether the p-axis has tick marks or a full grid.

    If :data:`p_ticks_major` is non-zero, then if p_grid is False tick marks
    will be displayed at every major tick. If p_grid is True, instead of ticks,
    a vertical line will be displayed at every major tick.

    :data:`p_grid` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    p_grid_label = BooleanProperty(True)
    """Whether labels should be displayed beneath each major tick. If true,
    each major tick will have a label containing the axis value.

    :data:`p_grid_label` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to False.
    """

    plabel = StringProperty("Power")
    """The label for the p-axis. If not empty it is displayed in the center of
    the axis.

    :data:`plabel` is a :class:`~kivy.properties.StringProperty`,
    defaults to ''.
    """

    hmin = NumericProperty(0.0)
    """The h-axis minimum value.

    If :data:`hlog` is True, hmin must be larger than zero.

    :data:`hmin` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    hmax = NumericProperty(100.0)
    """The h-axis maximum value, larger than hmin.

    :data:`hmax` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    hlog = BooleanProperty(False)
    """Determines whether the h-axis should be displayed logarithmically (True)
    or linearly (False).

    :data:`hlog` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    h_ticks_major = BoundedNumericProperty(0, min=0)
    """Distance between major tick marks on the h-axis.

    Determines the distance between the major tick marks. Major tick marks
    start from min and re-occur at every ticks_major until :data:`hmax`.
    If :data:`hmax` doesn't overlap with a integer multiple of ticks_major,
    no tick will occur at :data:`hmax`. Zero indicates no tick marks.

    If :data:`hlog` is true, then this indicates the distance between ticks
    in multiples of current decade. E.g. if :data:`hmin` is 0.1 and
    ticks_major is 0.1, it means there will be a tick at every 10th of the
    decade, i.e. 0.1 ... 0.9, 1, 2... If it is 0.3, the ticks will occur at
    0.1, 0.3, 0.6, 0.9, 2, 5, 8, 10. You'll notice that it went from 8 to 10
    instead of to 20, that's so that we can say 0.5 and have ticks at every
    half decade, e.g. 0.1, 0.5, 1, 5, 10, 50... Similarly, if ticks_major is
    1.5, there will be ticks at 0.1, 5, 100, 5,000... Also notice, that there's
    always a major tick at the start. Finally, if e.g. :data:`hmin` is 0.6
    and this 0.5 there will be ticks at 0.6, 1, 5...

    :data:`h_ticks_major` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    h_ticks_minor = BoundedNumericProperty(0, min=0)
    """The number of sub-intervals that divide h_ticks_major.

    Determines the number of sub-intervals into which ticks_major is divided,
    if non-zero. The actual number of minor ticks between the major ticks is
    ticks_minor - 1. Only used if ticks_major is non-zero. If there's no major
    tick at hmax then the number of minor ticks after the last major
    tick will be however many ticks fit until hmax.

    If self.hlog is true, then this indicates the number of intervals the
    distance between major ticks is divided. The result is the number of
    multiples of decades between ticks. I.e. if ticks_minor is 10, then if
    ticks_major is 1, there will be ticks at 0.1, 0.2...0.9, 1, 2, 3... If
    ticks_major is 0.3, ticks will occur at 0.1, 0.12, 0.15, 0.18... Finally,
    as is common, if ticks major is 1, and ticks minor is 5, there will be
    ticks at 0.1, 0.2, 0.4... 0.8, 1, 2...

    :data:`h_ticks_minor` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    h_grid = BooleanProperty(False)
    """Determines whether the h-axis has tick marks or a full grid.

    If :data:`h_ticks_major` is non-zero, then if h_grid is False tick marks
    will be displayed at every major tick. If h_grid is True, instead of ticks,
    a vertical line will be displayed at every major tick.

    :data:`h_grid` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    h_grid_label = BooleanProperty(True)
    """Whether labels should be displayed beneath each major tick. If true,
    each major tick will have a label containing the axis value.

    :data:`h_grid_label` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to False.
    """

    hlabel = StringProperty("Count")
    """The label for the h-axis. If not empty it is displayed in the center of
    the axis.

    :data:`hlabel` is a :class:`~kivy.properties.StringProperty`,
    defaults to ''.
    """

    power_histogram_width = NumericProperty("0.25")
    power_colorbar_width = NumericProperty("0.02")
    average_power_height = NumericProperty("0.2")

    padding = NumericProperty("4dp")
    """Padding distances between the labels, axes titles and graph, as
    well between the widget and the objects near the boundaries.

    :data:`padding` is a :class:`~kivy.properties.NumericProperty`, defaults
    to 5dp.
    """

    font_size = NumericProperty("15sp")
    """Font size of the labels.

    :data:`font_size` is a :class:`~kivy.properties.NumericProperty`, defaults
    to 15sp.
    """

    x_ticks_angle = NumericProperty(0)
    """Rotate angle of the x-axis tick marks.

    :data:`x_ticks_angle` is a :class:`~kivy.properties.NumericProperty`,
    defaults to 0.
    """

    precision = StringProperty("%g")
    """Determines the numerical precision of the tick mark labels. This value
    governs how the numbers are converted into string representation. Accepted
    values are those listed in Python's manual in the
    "String Formatting Operations" section.

    :data:`precision` is a :class:`~kivy.properties.StringProperty`, defaults
    to '%g'.
    """

    draw_border = BooleanProperty(True)
    """Whether a border is drawn around the canvas of the graph where the
    plots are displayed.

    :data:`draw_border` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to True.
    """

    plots = ListProperty([])
    """Holds a list of all the plots in the graph. To add and remove plots
    from the graph use :data:`add_plot` and :data:`add_plot`. Do not add
    directly edit this list.

    :data:`plots` is a :class:`~kivy.properties.ListProperty`,
    defaults to [].
    """

    #    view_size = ObjectProperty((0, 0))
    """The size of the graph viewing area - the area where the plots are
    displayed, excluding labels etc.
    """

    #    view_pos = ObjectProperty((0, 0))
    """The pos of the graph viewing area - the area where the plots are
    displayed, excluding labels etc. It is relative to the graph's pos.
    """

    def get_histogram(self, image, maxsize=2000):
        step = (
            int(numpy.ceil(image.shape[0] / maxsize)),
            int(numpy.ceil(image.shape[1] / maxsize)),
        )
        if numpy.isscalar(step):
            step = (step, step)
        step_data = image[:: step[0], :: step[1]]
        step_data = step_data[numpy.isfinite(step_data)]
        hist = numpy.histogram(step_data, bins=500)

        return hist[1][:-1], hist[0]

    def set_image(self, image, offset):
        shape = image.shape
        if (
            self.valid_power_range[0] is not None
            and self.valid_power_range[1] > self.valid_power_range[0]
        ):
            vmin, vmax = self.valid_power_range
        else:
            vmin, vmax = image.min(), image.max()

        # get histogram
        hist = self.get_histogram(image)

        # get time-wise power
        if (
            self.valid_freq_range[0] is not None
            and self.valid_freq_range[1] > self.valid_freq_range[0]
        ):
            tpower = numpy.average(
                image[:, int(self.valid_freq_range[0]) : int(self.valid_freq_range[1])],
                axis=1,
            )
        else:
            tpower = numpy.average(image, axis=1)
        tmax = numpy.max(tpower)

        # set spectrogam
        gs_cursor = numpy.zeros(shape).T
        if self.valid_freq_range[0] is not None:
            gs_cursor[
                int(self.valid_freq_range[0]) : int(self.valid_freq_range[1]), :
            ] = 1
        gs_cursor = numpy.dstack(
            (gs_cursor * 128, gs_cursor * 128, gs_cursor * 64)
        ).astype(numpy.uint8)
        gs_image = numpy.maximum(
            0, numpy.minimum(255, ((image.T - vmin) / (vmax - vmin) * 255))
        ).astype(numpy.uint8)
        gs_image = numpy.dstack((gs_image, gs_image, gs_image))

        gs_image = numpy.maximum(gs_image, gs_cursor)

        texture = Texture.create(size=shape)
        texture.blit_buffer(
            pbuffer=gs_image.tobytes(), colorfmt="bgr", bufferfmt="ubyte"
        )
        self._spectrogram_area.texture = texture

        self.x_ticks_major = self.get_safe_interval(tpower.shape[0], 4)
        self.xmin = offset
        self.xmax = offset + tpower.shape[0]
        self.ymin = 0
        self.ymax = float(max(self.ymax, tmax))
        self.fmin = 0
        self.fmax = image.shape[1]
        self.pmin = 0
        self.pmax = float(numpy.max(hist[0]))
        self.hmin = 0
        self.hmax = float(numpy.max(hist[1]))

        self.average_power_plot.points = enumerate(tpower, offset)
        self.power_histogram_plot.points = zip(hist[1], hist[0])

    def get_safe_interval(self, length, count):
        length = max(length, 1)
        cr = float(length) / count
        cl = log10(cr)
        cli = floor(cl)
        clr = cl - cli

        if clr < log10(2):
            return 10 ** cli
        elif clr < log10(5):
            return 2 * 10 ** cli
        else:
            return 5 * 10 ** cli


# end of file
