#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/libharkio3/position.py

"""

import xml.etree.ElementTree as ET
from xml.etree.ElementTree import Element

import numpy

from .defs import CoordinateSystem, UseTagType
from .error import LibHarkIOError


import pydantic

class Position(pydantic.BaseModel):
    class Config:
        arbitrary_types_allowed = True

    position_id:int = 0
    coordinate_system: CoordinateSystem = CoordinateSystem.Cartesian,
    coordinate: list[float] = [0, 0, 0]
    path: str=None
    channels_use: UseTagType = UseTagType.NON_TAG
    channels: list[int] = []
    tfcalc_params: dict = {}

    def as_coordinate(self, coordinate_system: CoordinateSystem) -> tuple[float, float, float]:
        if coordinate_system == CoordinateSystem.Cartesian:

            if self.coordinate_system == CoordinateSystem.Cartesian:
                return self.coordinate
        
            elif self.coordinate_system == CoordinateSystem.Polar:
            
                azimuth, elevation, radius = self.coordinate

                azimuth = numpy.deg2rad(azimuth)
                elevation = numpy.deg2rad(elevation)

                coef = radius * numpy.cos(elevation)
                return (coef * numpy.cos(azimuth),
                        coef * numpy.sin(azimuth),
                        radius * numpy.sin(elevation))

            else:
                raise LibHarkIOError(f'Unexpected coordinate system {self.coordinate_system}')

        elif coordinate_system == CoordinateSystem.Polar:
            raise LibHarkIOError(f'Unsupported coordinate system {coordinate_system}')

        else:
            raise LibHarkIOError(f'Unexpected coordinate system {coordinate_system}')


class PositionParser:
    @classmethod
    def as_element(cls, position: Position) -> Element:

        def _format(value):
            return f'{value:.4f}'

        if position.coordinate_system == CoordinateSystem.Cartesian:
            attrs = dict(
                x=_format(position.coordinate[0]),
                y=_format(position.coordinate[1]),
                z=_format(position.coordinate[2]),
            )
        elif position.coordinate_system == CoordinateSystem.Polar:
            attrs = dict(
                azimuth=_format(position.coordinate[0]),
                elevation=_format(position.coordinate[1]),
                z=_format(position.coordinate[2]),
            )
        else:
            raise LibHarkIOError(f'Unexpected coordinate system {position.coordinate_system}')

        attrs.update(dict(
            path=str(position.path),
            id=str(position.position_id),
        ))

        element = ET.Element('position', attrs)
        return element

    @classmethod
    def from_element(cls, element: Element, **kwargs) -> Position:
        if element.tag != 'position':
            raise LibHarkIOError()

        coordinate_system = kwargs.get(
            'coordinate_system', CoordinateSystem.Cartesian)

        position_id = None
        coordinate = [0., 0., 0.]
        path = None
        for akey, aval in element.attrib.items():

            if akey.lower() == 'x':
                coordinate[0] = float(aval)

            elif akey.lower() == 'y':
                coordinate[1] = float(aval)

            elif akey.lower() == 'z':
                coordinate[2] = float(aval)

            elif akey.lower() == 'azimuth':
                coordinate[0] = float(aval)

            elif akey.lower() == 'elevation':
                coordinate[1] = float(aval)

            elif akey.lower() == 'z':
                coordinate[2] = float(aval)

            elif akey.lower() == 'path':

                if path is None:
                    path = aval
                else:
                    raise LibHarkIOError(f'Duplicate path in <{element.tag}>')

            elif akey.lower() == 'id':
                position_id = int(aval)
        
        kwargs = {}
        if path:
            kwargs['path'] = path

        return Position(position_id=position_id, coordinate_system=coordinate_system, coordinate=coordinate, **kwargs)

# end of file
