#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/app/validxml.py

"""

import argparse
import collections
import json
import sys

from .. import utils
from ..libharkio3 import Config, HarkXML, HarkXMLParser
from .defs import PositionType
from .error import HarkToolError


def parse_args(args):
    parser = argparse.ArgumentParser(
        description='Consistency check of xml file.')

    parser.add_argument('--mic', '-m', action='store_true',
                        dest='isMic', default=False, help='Check for microphone positions')
    parser.add_argument('--src', '-s', action='store_true', dest='isSrc',
                        default=False, help='Check for source (soundsource, tsp, impulse) positions')
    parser.add_argument('--json', '-j', action='store_true',
                        dest='isJson', default=False, help='Show info (JSON)')
    parser.add_argument('--logLevel', type=str, metavar='{E|W|I|D}', default='E',
                        required=False, help='Log information level. (default=%(default)s)', dest='logLevel')
    parser.add_argument(type=str, metavar='INFILE',
                        help='Input file name.', dest='input')

    return parser.parse_args(args)


def main(args=sys.argv[1:]):
    args = parse_args(args)

    logger = utils.initialize_logger(args)
    logger.info('Args: %s', args)

    xml = HarkXMLParser.from_file(args.input)

    if args.isMic:
        _validate_as_position_xml('MIC', xml)

    elif args.isSrc:
        _validate_as_position_xml('SRC', xml)

    else:
        _validate_harkxml(xml)

    if args.isJson:
        j = {}
        j['type'] = xml.positions.type

        if xml.positions.type == 'microphone':
            j['channels'] = [p.id for p in xml.positions]

        elif xml.positions[0].channels_use:
            j['channels'] = xml.positions[0].channels

        if xml.config is not None:
            config: Config = xml.config
            j['config'] = {
                'comment': config.comment,
                'syncadd': config.syncadd,
                'orgtsppath': config.orttsppath,
                'tspoffset': config.tspoffset,
                'peakfrom': config.peaks[0],
                'peakto': config.peaks[1],
                'fftlen': config.nfft,
                'samplingrate': config.samplingrate,
                'signalmax': config.signal_max,
                'tsplength': config.tsp_length,
            }

        json.dump(j, sys.stdout)


def _validate_as_position_xml(position_type: str, xml: HarkXML) -> None:
    message = ''

    if xml is None:
        message += '[hark_xml]'

    else:
        if xml.positions is None:
            message += '[positions]'

        else:
            positions = xml.positions
            if position_type == 'MIC':
                if positions.type != 'microphone':
                    message += f'[positions (type is not microphone): {positions.type})]'

                elif len(positions) <= 0:
                    message += '[position is empty]'

                else:
                    for ipos, pos in enumerate(positions):
                        # if pos.id >= MAX_CH:
                        #     message += '[position (id is not less than %d: %d)]'.format(
                        #         MAX_CH, pos.id)

                        for ipos2, pos2 in enumerate(positions):
                            if ipos == ipos2:
                                continue
                            elif pos.id == pos2.id:
                                message += f'[position (id is not unique: {pos.id})]'

            elif position_type == 'SRC':
                if positions.type not in ('soundsource', 'tsp', 'impulse'):
                    message += f'[positions (type is not soundsource|tsp|impulse: {positions.type})]'

            else:
                raise HarkToolError(f'Illegal type {position_type}')

            id_counts = collections.Counter(p.id for p in positions)
            id_dups = [k for k, v in id_counts.items() if v > 1]
            if len(id_dups) > 0:
                message += f'[position (id is not unique: {id_dups})]'

    if len(message) == 0:
        print('SUCCESS', file=sys.stderr)

    else:
        print('Consistency error', file=sys.stderr)
        print(message, file=sys.stderr)


def _validate_harkxml(xml: HarkXML) -> None:
    message = ''

    if xml is None:
        message += '[hark_xml]'

    else:
        if xml.positions is None:
            message += '[positions]'

        else:
            positions = xml.positions

            poses_case = {
                'soundsource': PositionType.SRC,
                'tsp': PositionType.TSP,
                'impulse': PositionType.IMPULSE,
                'noise': PositionType.NOISE,
                'microphone': PositionType.MIC,
            }.get(positions.type, None)

            if poses_case is None:
                message += f'[positions (unknown type: {positions.type})]'

            if poses_case in (PositionType.SRC, PositionType.TSP, PositionType.IMPULSE):

                if xml.config is None:
                    message += '[config]'
                else:
                    config = xml.config
                    if config.comment is None:
                        message += '[comment]'
                    if config.synchronousAverage < 0:
                        message += '[SynchronousAverage]'
                    if config.tsp_path is None:
                        message += '[TSPpath]'
                    if config.tsp_offset < 0:
                        message += '[TSPOffset]'
                    if config.peak_search[0] < 0 or config.peak_search[1] < 0:
                        message += '[PeakSearch]'
                    if config.nfft <= 0:
                        message += '[nfft]'
                    if config.samplingRate <= 0:
                        message += '[samplingRate]'
                    if config.signal_max <= 0:
                        message += '[signalMax]'
                    if config.tsp_length <= 0:
                        message += '[TSPLength]'

            if poses_case in (PositionType.SRC, PositionType.TSP,
                              PositionType.IMPULSE, PositionType.NOISE, PositionType.MIC):
                if len(positions) <= 0:
                    message += '[position]'

                if xml.neighbors is None:
                    message += '[neighbors]'

                elif xml.neighbors <= 0:
                    message += '[neighbor]'

    if len(message) == 0:
        print('SUCCESS', file=sys.stderr)

    else:
        print('Consistency error', file=sys.stderr)
        print(message, file=sys.stderr)


# end of file
