import argparse
import enum
import json
import sys
import types
import typing

import numpy

from .. import utils
from ..libharkio3 import (Config, Positions, TransferFunction,
                          TransferFunctionParser)
from .defs import LogLevel, TFType
from .error import HarkToolError
from .utils import StoreLogLevelAction

_default_values = types.SimpleNamespace(
    show_as_text=False,
    show_as_json=False,
    log_level=LogLevel.Error,
)


def parse_args(args):
    parser = argparse.ArgumentParser(
        description='Consistency check of the transfer function.')

    parser.add_argument('--show', '-s', required=False,
                        dest='show_as_text', action='store_true',
                        default=_default_values.show_as_text,
                        help='Show details')
    parser.add_argument('--json', '-j', required=False,
                        dest='show_as_json', action='store_true',
                        default=_default_values.show_as_json,
                        help='Show details (JSON)')
    parser.add_argument('--log-level', '--llevel', metavar='{E|W|I|D}', required=False,
                        type=LogLevel, action=StoreLogLevelAction, dest='log_level',
                        default=_default_values.log_level,
                        help='Log information level. (default=%(default)s)', )
    parser.add_argument(metavar='INFILE',
                        type=str, dest='input_file',
                        help='Input zip file')
    return parser.parse_args(args)


def main(args=sys.argv[1:]):
    args = parse_args(args)

    logger = utils.initialize_logger(args)

    validate_tf(**vars(args), logger=logger)


def validate_tf(
    show_as_text: bool,
    show_as_json: bool,
    input_file: str,
    logger: typing.Any,
    **kwargs,
) -> None:

    # load tf
    tf: TransferFunction = TransferFunctionParser.from_zipfile(input_file)

    # show tf
    if show_as_text:
        print(tf)
    elif show_as_json:
        json.dump(tf.model_dump(), sys.stdout)

# end of file
