#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/app/gentsp.py

"""

import argparse
import sys
import types

import numpy

from .. import utils
from .defs import LogLevel
from .error import HarkToolError
from .utils import StoreLogLevelAction

_default_values = types.SimpleNamespace(
    tsp_count=16,
    tsp_offset=0,
    sampling_rate=16000,
    signal_max=None,
    log_level=LogLevel.Error,
)


def parse_args(args):
    parser = argparse.ArgumentParser(
        description='TSP signal generating tool. The file is saved as 32bit float wave.')
    parser.add_argument('--num', '-n', type=int, metavar='NUM', default=_default_values.tsp_count,
                        required=False, help='Number of TSPs. (default=%(default)s)', dest='tsp_count')
    parser.add_argument('--offset', '-o', type=int, metavar='NUM', default=_default_values.tsp_offset,
                        required=False, help='Offset of the TSP. (default=%(default)s)', dest='tsp_offset')
    parser.add_argument('--rate', '-r', type=int, metavar='NUM', default=_default_values.sampling_rate, required=False,
                        help='Sampling rate of TSP. (default=%(default)s)', dest='sampling_rate')
    parser.add_argument('--src', '-s', type=str, metavar='PATH',
                        required=True, help='Input tsp raw (32 bit float) file.', dest='original_tsp_file')
    parser.add_argument('--volume', '-v', type=float, metavar='NUM', default=_default_values.signal_max,
                        required=False, help='Max amplitude for normalization.', dest='signal_max')
    parser.add_argument('--logLevel', type=LogLevel, action=StoreLogLevelAction, metavar='{E|W|I|D}', default=_default_values.log_level,
                        required=False, help='Log information level. (default=%(default)s)', dest='log_level')
    parser.add_argument(type=str, metavar='PATH',
                        help='Output file name.', dest='output_file')

    return parser.parse_args(args)


def main(args=sys.argv[1:]):
    args = parse_args(args)

    logger = utils.initialize_logger(args)

    logger.debug('n: %s', args.tsp_count)
    logger.debug('o: %s', args.tsp_offset)
    logger.debug('r: %s', args.sampling_rate)
    logger.debug('s: %s', args.original_tsp_file)
    logger.debug('v: %s', args.signal_max)

    try:
        origtsp, sr = utils.TSP.load(args.original_tsp_file)
    except BaseException as ex:
        raise HarkToolError('Failed to read TSP file.') from ex

    if sr != args.samplingRate:
        raise HarkToolError(
            'Sampling rate mismatch: expected={}, actual={}'.format(args.sampling_rate, sr))

    maxamp = numpy.abs(origtsp).max()
    if args.signal_max is None:
        if maxamp > 1.0:
            logger.warning(
                "Max amplitude is over 1.0. Output wave file may be clipped.")

    else:
        origtsp *= 1.0 / maxamp * args.signal_max

    gentsp = numpy.r_[
        numpy.zeros((args.tsp_offset,) +
                    origtsp.shape[1:], dtype=origtsp.dtype),
        numpy.concatenate([origtsp] * args.tsp_count, axis=0),
    ]

    try:
        utils.TSP.save(gentsp, args.sampling_rate, args.output_file)
    except BaseException as ex:
        raise HarkToolError('Failed to write TSP data.') from ex

    logger.debug("File generated : %s", args.output_file)

    logger.info("SUCCESS")
    logger.info("output file : %s", args.output_file)

# end of file
