#!/usr/bin/python
# -*- coding: utf-8 -*-
"""

harktool/app/genneighbors.py

"""

import argparse
import sys
import types

from .. import utils
from ..libharkio3 import HarkXMLParser, Neighbors
from ..libharkio3.defs import NeighborAlgorithm
from .defs import LogLevel
from .utils import StoreLogLevelAction, StoreNeighborAlgorithmTypeAction

_default_values = types.SimpleNamespace(
    algorithm = NeighborAlgorithm.NearestNeighbor,
    nearst = 3,
    threshold = 0.5,
    log_level = LogLevel.Error,
)

def parse_args(args):
    parser = argparse.ArgumentParser(
        description='The command creates an xml file that contains neighbors tag of xml file.')

    parser.add_argument('--algorithm', '-a', type=NeighborAlgorithm, action=StoreNeighborAlgorithmTypeAction, metavar='NUM', dest='algorithm',
                        default=_default_values.algorithm, help='Algorithms (1:NearestNeighbor) (default=%(default)s)')
    parser.add_argument('--neaest', '-n', type=int, metavar='NUM', dest='nearest',
                        default=_default_values.nearst, help='Maximum number of neighbors (default=%(default)s)')
    parser.add_argument('--threshold', '-t', type=float, metavar='NUM', dest='threshold',
                        default=_default_values.threshold, help='Threshold for neighbor decision. (default=%(default)s)')
    parser.add_argument('--logLevel', type=LogLevel, action=StoreLogLevelAction, metavar='{E|W|I|D}', default=_default_values.log_level,
                        required=False, help='Log information level. (default=%(default)s)', dest='log_level')
    parser.add_argument(type=str, metavar='INFILE',
                        help='Input file name.', dest='input_file')
    parser.add_argument(type=str, metavar='OUTFILE',
                        help='Output file name.', dest='output_file')

    return parser.parse_args(args)


def main(args=sys.argv[1:]):
    args = parse_args(args)

    logger = utils.initialize_logger(args)
    logger.info('Args: %s', args)
    
    # make positions
    xml = HarkXMLParser.from_file(args.input_file)
    
    neighbors = Neighbors()
    neighbors.calculate_neighbors(positions=xml.positions,
        algorithm=args.algorithm, count=args.nearest, threshold=args.threshold)
    xml.neighbors = neighbors

    HarkXMLParser.as_file(xml, args.output_file)

    logger.debug('XML file created')


# end of file
