#! /usr/bin/env python3
# -*- coding: utf-8 -*-

import pickle
import queue
import sys
import threading
from decimal import Decimal
from math import ceil, floor

import kivy
import numpy
from kivy import metrics
from kivy.app import App
from kivy.clock import Clock
from kivy.config import Config
from kivy.core.window import Window
from kivy.graphics import Color, Fbo, Line, Mesh, Rectangle, RenderContext
from kivy.graphics.texture import Texture
from kivy.lang import Builder

# from kivy.properties import ObjectProperty
from kivy.properties import (
    AliasProperty,
    BooleanProperty,
    BoundedNumericProperty,
    DictProperty,
    ListProperty,
    NumericProperty,
    ObjectProperty,
    StringProperty,
)
from kivy.uix.boxlayout import BoxLayout
from kivy.uix.gridlayout import GridLayout
from kivy.uix.label import Label
from kivy.uix.stencilview import StencilView
from kivy.uix.widget import Widget
from kivy.utils import get_color_from_hex as rgb
from kivy_garden.graph import Graph, LinePlot

kivy.require("1.8.0")
Config.set("kivy", "log_level", "error")
Config.set("kivy", "log_enable", "0")
Config.write()

kv = """
#:import Graph kivy_garden.graph.Graph
<KivtyWaveformCodeGraphs>:
    x_ticks_major: 2000
    x_ticks_minor: 10
    x_grid: True
    x_grid_label: True
    y_grid: True
    y_grid_label: True
    ymin: -1.
    ymax:  1.
"""

Builder.load_string(kv)

BUF_SIZE = 10
q = queue.Queue(BUF_SIZE)


"""
class KivyWaveformCodeGraph(Graph):
    def __init__(self):
        super(KivyWaveformCodeGraph, self).__init__()
        self.x_grid_label = True

    def _get_ticks(self, major, minor, s_min, s_max):
        x = super(KivyWaveformCodeGraph, self)._get_ticks(major, minor, s_min, s_max)
        # print((major, minor, s_min, s_max))
        # print(x)
        return x
"""


Builder.load_string(
    """
<GraphRotatedLabel>:
    canvas.before:
        PushMatrix
        Rotate:
            angle: root.angle
            axis: 0, 0, 1
            origin: root.center
    canvas.after:
        PopMatrix
"""
)


class GraphRotatedLabel(Label):
    angle = NumericProperty(0)


class MultiGraph(Widget):
    """Graph class, see module documentation for more information."""

    # triggers a full reload of graphics
    _trigger = ObjectProperty(None)
    # triggers only a repositioning of objects due to size/pos updates
    _trigger_size = ObjectProperty(None)
    # triggers only a update of colors, e.g. tick_color
    _trigger_color = ObjectProperty(None)
    # holds widget with the x-axis label
    _xlabel = ObjectProperty(None)
    # holds widget with the y-axis label
    _ylabel = ObjectProperty(None)
    # holds all the x-axis tick mark labels
    _x_grid_label = ListProperty([])
    # holds all the y-axis tick mark labels
    _y_grid_label = ListProperty([])
    # the mesh drawing all the ticks/grids
    _mesh_ticks = ObjectProperty(None)
    # the mesh which draws the surrounding rectangle
    _mesh_rects = ObjectProperty(None)
    # a list of locations of major and minor ticks. The values are not
    # but is in the axis min - max range
    _ticks_majorx = ListProperty([])
    _ticks_minorx = ListProperty([])
    _ticks_majory = ListProperty([])
    _ticks_minory = ListProperty([])

    tick_color = ListProperty([0.25, 0.25, 0.25, 1])
    """Color of the grid/ticks, default to 1/4. grey.
    """

    background_color = ListProperty([0, 0, 0, 0])
    """Color of the background, defaults to transparent
    """

    border_color = ListProperty([1, 1, 1, 1])
    """Color of the border, defaults to white
    """

    label_options = DictProperty()
    """Label options that will be passed to `:class:`kivy.uix.Label`.
    """

    _with_stencilbuffer = BooleanProperty(True)
    """Whether :class:`Graph`'s FBO should use FrameBuffer (True) or not (False).

    .. warning:: This property is internal and so should be used with care. It can break
    some other graphic instructions used by the :class:`Graph`, for example you can have
    problems when drawing :class:`SmoothLinePlot` plots, so use it only when you know
    what exactly you are doing.

    :data:`_with_stencilbuffer` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to True.
    """

    def __init__(self, **kwargs):
        super().__init__(**kwargs)

        with self.canvas:
            self._fbo = Fbo(size=self.size, with_stencilbuffer=self._with_stencilbuffer)

        with self._fbo:
            self._background_color = Color(*self.background_color)
            self._background_rect = Rectangle(size=self.size)
            self._mesh_ticks_color = Color(*self.tick_color)
            self._mesh_ticks = Mesh(mode="lines")
            self._mesh_rect_color = Color(*self.border_color)
            self._mesh_rects = []  # = Mesh(mode='line_strip')

        with self.canvas:
            Color(1, 1, 1)
            self._fbo_rect = Rectangle(size=self.size, texture=self._fbo.texture)

        self._plot_area = StencilView()
        self.add_widget(self._plot_area)

        t = self._trigger = Clock.create_trigger(self._redraw_all)
        ts = self._trigger_size = Clock.create_trigger(self._redraw_size)
        tc = self._trigger_color = Clock.create_trigger(self._update_colors)

        self.bind(
            center=ts,
            padding=ts,
            precision=ts,
            plots=ts,
            x_grid=ts,
            y_grid=ts,
            draw_border=ts,
        )
        self.bind(
            xmin=t,
            xmax=t,
            x_ticks_major=t,
            x_ticks_minor=t,
            xlabel=t,
            x_grid_label=t,
            ymin=t,
            ymax=t,
            y_ticks_major=t,
            y_ticks_minor=t,
            ylabel=t,
            y_grid_label=t,
            font_size=t,
            label_options=t,
            x_ticks_angle=t,
        )
        self.bind(tick_color=tc, background_color=tc, border_color=tc)
        self._trigger()

    def add_widget(self, widget):
        if widget is self._plot_area:
            canvas = self.canvas
            self.canvas = self._fbo
        super().add_widget(widget)
        if widget is self._plot_area:
            self.canvas = canvas

    def remove_widget(self, widget):
        if widget is self._plot_area:
            canvas = self.canvas
            self.canvas = self._fbo
        super().remove_widget(widget)
        if widget is self._plot_area:
            self.canvas = canvas

    def _get_ticks(self, major, minor, s_min, s_max):
        if major and s_max > s_min:
            # distance between each tick
            tick_dist = major / float(minor if minor else 1.0)
            n_ticks = int(floor((s_max - s_min) / tick_dist) + 1)
            points_major = [0] * int(floor((s_max - s_min) / float(major)) + 1)
            points_minor = [0] * (n_ticks - len(points_major) + 1)
            k = 0  # position in points major
            k2 = 0  # position in points minor
            for m in range(0, n_ticks):
                if minor and m % minor:
                    points_minor[k2] = m * tick_dist + s_min
                    k2 += 1
                else:
                    points_major[k] = m * tick_dist + s_min
                    k += 1
            del points_major[k:]
            del points_minor[k2:]
        else:
            points_major = []
            points_minor = []
        return points_major, points_minor

    def _update_labels(self):
        xlabel = self._xlabel
        ylabel = self._ylabel
        x = self.x
        y = self.y
        width = self.width
        height = self.height
        padding = self.padding
        x_next = padding + x
        y_next = padding + y
        xextent = width + x
        yextent = height + y
        ymin = self.ymin
        ymax = self.ymax
        xmin = self.xmin
        precision = self.precision
        # set up x and y axis labels
        if xlabel:
            xlabel.text = self.xlabel
            xlabel.texture_update()
            xlabel.size = xlabel.texture_size
            xlabel.pos = int(x + width / 2.0 - xlabel.width / 2.0), int(padding + y)
            y_next += padding + xlabel.height
        if ylabel:
            ylabel.text = self.ylabel
            ylabel.texture_update()
            ylabel.size = ylabel.texture_size
            ylabel.x = padding + x - (ylabel.width / 2.0 - ylabel.height / 2.0)
            x_next += padding + ylabel.height
        xpoints = self._ticks_majorx
        xlabels = self._x_grid_label
        xlabel_grid = self.x_grid_label
        ylabel_grid = self.y_grid_label
        ypoints = self._ticks_majory
        ylabels = self._y_grid_label
        # now x and y tick mark labels
        if len(ylabels) and ylabel_grid:
            # horizontal size of the largest tick label, to have enough room
            ylabels[0].text = precision % float(ypoints[0])
            ylabels[0].texture_update()
            y1 = ylabels[0].texture_size
            y_start = (
                y_next
                + (padding + y1[1] if len(xlabels) and xlabel_grid else 0)
                + (padding + y1[1] if not y_next else 0)
            )
            yextent = y + height - padding - y1[1] / 2.0
            ratio = (yextent - y_start) / (ymax - ymin)
            y_start -= y1[1] / 2.0
            y1 = y1[0]
            for k in range(len(ylabels)):
                ylabels[k].text = precision % float(ypoints[k])
                ylabels[k].texture_update()
                ylabels[k].size = ylabels[k].texture_size
                y1 = max(y1, ylabels[k].texture_size[0])
                ylabels[k].pos = (
                    int(x_next),
                    int(y_start + (ypoints[k] - ymin) * ratio),
                )
            if len(ylabels) > 1 and ylabels[0].top > ylabels[1].y:
                pass
            else:
                x_next += y1 + padding
        if len(xlabels) and xlabel_grid:
            # find the distance from the end that'll fit the last tick label
            xlabels[0].text = precision % float(xpoints[-1])
            xlabels[0].texture_update()
            xextent = x + width - xlabels[0].texture_size[0] / 2.0 - padding
            # find the distance from the start that'll fit the first tick label
            if not x_next:
                xlabels[0].text = precision % float(xpoints[0])
                xlabels[0].texture_update()
                x_next = padding + xlabels[0].texture_size[0] / 2.0
            ratio = (xextent - x_next) / (self.xmax - xmin)
            right = -1
            for k in range(len(xlabels)):
                xlabels[k].text = precision % float(xpoints[k])
                # update the size so we can center the labels on ticks
                xlabels[k].texture_update()
                xlabels[k].size = xlabels[k].texture_size
                half_ts = xlabels[k].texture_size[0] / 2.0
                xlabels[k].pos = (
                    int(x_next + (xpoints[k] - xmin) * ratio - half_ts),
                    int(y_next),
                )
                if xlabels[k].x < right:
                    break
                right = xlabels[k].right
            y_next += padding + xlabels[0].texture_size[1]
        # now re-center the x and y axis labels
        if xlabel:
            xlabel.x = int(x_next + (xextent - x_next) / 2.0 - xlabel.width / 2.0)
        if ylabel:
            ylabel.y = int(y_next + (yextent - y_next) / 2.0 - ylabel.height / 2.0)
            ylabel.angle = 45
        return x_next - x, y_next - y, xextent - x, yextent - y

    def _update_ticks(self, size):
        # re-compute the positions of the bounding rectangle
        h = size[3] - size[1]
        nplots = len(self.plots)
        if self.draw_border:
            for i in range(nplots):
                mesh = self._mesh_rects[i]
                vert = mesh.vertices

                s0 = size[0]
                s1 = (
                    size[1]
                    + (h - self.padding * (nplots - 1)) * i / nplots
                    + self.padding * i
                )
                s2 = size[2]
                s3 = (
                    size[1]
                    + (h - self.padding * (nplots - 1)) * (i + 1) / nplots
                    + self.padding * i
                )

                vert[0] = s0
                vert[1] = s1
                vert[4] = s2
                vert[5] = s1
                vert[8] = s2
                vert[9] = s3
                vert[12] = s0
                vert[13] = s3
                vert[16] = s0
                vert[17] = s1
                mesh.vertices = vert

        else:
            for i in range(nplots):
                mesh = self._mesh_rects[i]
                vert = mesh.vertices
                vert[0:18] = [0 for k in range(18)]
                mesh.vertices = vert

        # re-compute the positions of the x/y axis ticks
        mesh = self._mesh_ticks
        vert = mesh.vertices
        start = 0
        xpoints = self._ticks_majorx
        ypoints = self._ticks_majory
        xpoints2 = self._ticks_minorx
        ypoints2 = self._ticks_minory
        xmin = self.xmin
        xmax = self.xmax
        nplots = len(self.plots)
        ymin = self.ymin
        ymax = self.ymax

        for _i in range(nplots):
            if len(xpoints):
                top = size[3] if self.x_grid else metrics.dp(12) + size[1]
                ratio = (size[2] - size[0]) / (xmax - xmin)
                for k in range(start, len(xpoints) + start):
                    vert[k * 8] = size[0] + (xpoints[k - start] - xmin) * ratio
                    vert[k * 8 + 1] = size[1]
                    vert[k * 8 + 4] = vert[k * 8]
                    vert[k * 8 + 5] = top
                start += len(xpoints)
            if len(xpoints2):
                top = metrics.dp(8) + size[1]
                ratio = (size[2] - size[0]) / (xmax - xmin)
                for k in range(start, len(xpoints2) + start):
                    vert[k * 8] = size[0] + (xpoints2[k - start] - xmin) * ratio
                    vert[k * 8 + 1] = size[1]
                    vert[k * 8 + 4] = vert[k * 8]
                    vert[k * 8 + 5] = top
                start += len(xpoints2)
            if len(ypoints):
                top = size[2] if self.y_grid else metrics.dp(12) + size[0]
                ratio = (size[3] - size[1]) / (ymax - ymin)
                for k in range(start, len(ypoints) + start):
                    vert[k * 8 + 1] = size[1] + (ypoints[k - start] - ymin) * ratio
                    vert[k * 8 + 5] = vert[k * 8 + 1]
                    vert[k * 8] = size[0]
                    vert[k * 8 + 4] = top
                start += len(ypoints)
            if len(ypoints2):
                top = metrics.dp(8) + size[0]
                ratio = (size[3] - size[1]) / (ymax - ymin)
                for k in range(start, len(ypoints2) + start):
                    vert[k * 8 + 1] = size[1] + (ypoints2[k - start] - ymin) * ratio
                    vert[k * 8 + 5] = vert[k * 8 + 1]
                    vert[k * 8] = size[0]
                    vert[k * 8 + 4] = top
        mesh.vertices = vert

    x_axis = ListProperty([None])
    y_axis = ListProperty([None])

    def get_x_axis(self, axis=0):
        if axis == 0:
            return (self.xmin, self.xmax)
        info = self.x_axis[axis]
        return (info["min"], info["max"])

    def get_y_axis(self, axis=0):
        if axis == 0:
            return (self.ymin, self.ymax)
        info = self.y_axis[axis]
        return (info["min"], info["max"])

    def add_x_axis(self, xmin, xmax):
        data = {"min": xmin, "max": xmax}
        self.x_axis.append(data)
        return data

    def add_y_axis(self, ymin, ymax):
        data = {"min": ymin, "max": ymax}
        self.y_axis.append(data)
        return data

    def _update_plots(self, size):
        h = size[3] - size[1]
        n = len(self.plots)
        for i, plot in enumerate(self.plots):
            ss = (
                size[0],
                size[1] + (h - self.padding * (n - 1)) * i / n + self.padding * i,
                size[2],
                size[1] + (h - self.padding * (n - 1)) * (i + 1) / n + self.padding * i,
            )

            (xmin, xmax) = self.get_x_axis(plot.x_axis)
            (ymin, ymax) = self.get_y_axis(plot.y_axis)
            plot._update(False, xmin, xmax, False, ymin, ymax, ss)

    def _update_colors(self, *args):
        self._mesh_ticks_color.rgba = tuple(self.tick_color)
        self._background_color.rgba = tuple(self.background_color)
        self._mesh_rect_color.rgba = tuple(self.border_color)

    def _redraw_all(self, *args):
        # add/remove all the required labels
        xpoints_major, xpoints_minor = self._redraw_x(*args)
        ypoints_major, ypoints_minor = self._redraw_y(*args)

        mesh = self._mesh_ticks
        n_points = (
            len(xpoints_major)
            + len(xpoints_minor)
            + len(ypoints_major)
            + len(ypoints_minor)
        ) * len(self.plots)
        mesh.vertices = [0] * (n_points * 8)
        mesh.indices = range(n_points * 2)
        self._redraw_size()

    def _redraw_x(self, *args):
        font_size = self.font_size
        if self.xlabel:
            xlabel = self._xlabel
            if not xlabel:
                xlabel = Label()
                self.add_widget(xlabel)
                self._xlabel = xlabel

            xlabel.font_size = font_size
            for k, v in self.label_options.items():
                setattr(xlabel, k, v)

        else:
            xlabel = self._xlabel
            if xlabel:
                self.remove_widget(xlabel)
                self._xlabel = None
        grids = self._x_grid_label
        xpoints_major, xpoints_minor = self._get_ticks(
            self.x_ticks_major, self.x_ticks_minor, self.xmin, self.xmax
        )
        self._ticks_majorx = xpoints_major
        self._ticks_minorx = xpoints_minor

        if not self.x_grid_label:
            n_labels = 0
        else:
            n_labels = len(xpoints_major)

        for k in range(n_labels, len(grids)):
            self.remove_widget(grids[k])
        del grids[n_labels:]

        grid_len = len(grids)
        grids.extend([None] * (n_labels - len(grids)))
        for k in range(grid_len, n_labels):
            grids[k] = GraphRotatedLabel(
                font_size=font_size, angle=self.x_ticks_angle, **self.label_options
            )
            self.add_widget(grids[k])
        return xpoints_major, xpoints_minor

    def _redraw_y(self, *args):
        font_size = self.font_size
        if self.ylabel:
            ylabel = self._ylabel
            if not ylabel:
                ylabel = GraphRotatedLabel()
                self.add_widget(ylabel)
                self._ylabel = ylabel

            ylabel.font_size = font_size
            for k, v in self.label_options.items():
                setattr(ylabel, k, v)
        else:
            ylabel = self._ylabel
            if ylabel:
                self.remove_widget(ylabel)
                self._ylabel = None
        grids = self._y_grid_label
        ypoints_major, ypoints_minor = self._get_ticks(
            self.y_ticks_major, self.y_ticks_minor, self.ymin, self.ymax
        )
        self._ticks_majory = ypoints_major
        self._ticks_minory = ypoints_minor

        if not self.y_grid_label:
            n_labels = 0
        else:
            n_labels = len(ypoints_major)

        for k in range(n_labels, len(grids)):
            self.remove_widget(grids[k])
        del grids[n_labels:]

        grid_len = len(grids)
        grids.extend([None] * (n_labels - len(grids)))
        for k in range(grid_len, n_labels):
            grids[k] = Label(font_size=font_size, **self.label_options)
            self.add_widget(grids[k])
        return ypoints_major, ypoints_minor

    def _redraw_size(self, *args):
        # size a 4-tuple describing the bounding box in which we can draw
        # graphs, it's (x0, y0, x1, y1), which correspond with the bottom left
        # and top right corner locations, respectively
        self._clear_buffer()
        size = self._update_labels()
        self.view_pos = self._plot_area.pos = (size[0], size[1])
        self.view_size = self._plot_area.size = (size[2] - size[0], size[3] - size[1])

        nplots = len(self.plots)
        for i in range(nplots):
            mesh = self._mesh_rects[i]
            mesh.vertices = [0] * (5 * 4)
            mesh.indices = range(5)

        if self.size[0] and self.size[1]:
            self._fbo.size = self.size
        else:
            self._fbo.size = 1, 1  # gl errors otherwise
        self._fbo_rect.texture = self._fbo.texture
        self._fbo_rect.size = self.size
        self._fbo_rect.pos = self.pos
        self._background_rect.size = self.size
        self._update_ticks(size)
        self._update_plots(size)

    def _clear_buffer(self, *largs):
        fbo = self._fbo
        fbo.bind()
        fbo.clear_buffer()
        fbo.release()

    def add_plot(self, plot, channel=0):
        """Add a new plot to this graph.

        :Parameters:
            `plot`:
                Plot to add to this graph.

        >>> graph = Graph()
        >>> plot = MeshLinePlot(mode='line_strip', color=[1, 0, 0, 1])
        >>> plot.points = [(x / 10., sin(x / 50.)) for x in range(-0, 101)]
        >>> graph.add_plot(plot)
        """
        if plot in self.plots:
            return

        with self._fbo:
            Color(*self.border_color)
            mesh = Mesh(mode="line_strip")
            mesh.vertices = [0] * (5 * 4)
            mesh.indices = range(5)
            self._mesh_rects.append(mesh)

        add = self._plot_area.canvas.add
        for instr in plot.get_drawings():
            add(instr)
        plot.bind(on_clear_plot=self._clear_buffer)
        self.plots.append(plot)
        self._redraw_all()

    def remove_plot(self, plot, channel=0):
        """Remove a plot from this graph.

        :Parameters:
            `plot`:
                Plot to remove from this graph.

        >>> graph = Graph()
        >>> plot = MeshLinePlot(mode='line_strip', color=[1, 0, 0, 1])
        >>> plot.points = [(x / 10., sin(x / 50.)) for x in range(-0, 101)]
        >>> graph.add_plot(plot)
        >>> graph.remove_plot(plot)
        """
        if plot not in self.plots:
            return
        remove = self._plot_area.canvas.remove
        for instr in plot.get_drawings():
            remove(instr)
        plot.unbind(on_clear_plot=self._clear_buffer)
        self.plots.remove(plot)
        self._clear_buffer()

    def collide_plot(self, x, y):
        """Determine if the given coordinates fall inside the plot area. Use
        `x, y = self.to_widget(x, y, relative=True)` to first convert into
        widget coordinates if it's in window coordinates because it's assumed
        to be given in local widget coordinates, relative to the graph's pos.

        :Parameters:
            `x, y`:
                The coordinates to test.
        """
        adj_x, adj_y = x - self._plot_area.pos[0], y - self._plot_area.pos[1]
        return (
            0 <= adj_x <= self._plot_area.size[0]
            and 0 <= adj_y <= self._plot_area.size[1]
        )

    def to_data(self, x, y):
        """Convert widget coords to data coords. Use
        `x, y = self.to_widget(x, y, relative=True)` to first convert into
        widget coordinates if it's in window coordinates because it's assumed
        to be given in local widget coordinates, relative to the graph's pos.

        :Parameters:
            `x, y`:
                The coordinates to convert.

        If the graph has multiple axes, use :class:`Plot.unproject` instead.
        """
        adj_x = float(x - self._plot_area.pos[0])
        adj_y = float(y - self._plot_area.pos[1])
        norm_x = adj_x / self._plot_area.size[0]
        norm_y = adj_y / self._plot_area.size[1]
        conv_x = norm_x * (self.xmax - self.xmin) + self.xmin
        conv_y = norm_y * (self.ymax - self.ymin) + self.ymin
        return [conv_x, conv_y]

    xmin = NumericProperty(0.0)
    """The x-axis minimum value.

    :data:`xmin` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    xmax = NumericProperty(100.0)
    """The x-axis maximum value, larger than xmin.

    :data:`xmax` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    x_ticks_major = BoundedNumericProperty(0, min=0)
    """Distance between major tick marks on the x-axis.

    Determines the distance between the major tick marks. Major tick marks
    start from min and re-occur at every ticks_major until :data:`xmax`.
    If :data:`xmax` doesn't overlap with a integer multiple of ticks_major,
    no tick will occur at :data:`xmax`. Zero indicates no tick marks.

    :data:`x_ticks_major` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    x_ticks_minor = BoundedNumericProperty(0, min=0)
    """The number of sub-intervals that divide x_ticks_major.

    Determines the number of sub-intervals into which ticks_major is divided,
    if non-zero. The actual number of minor ticks between the major ticks is
    ticks_minor - 1. Only used if ticks_major is non-zero. If there's no major
    tick at xmax then the number of minor ticks after the last major
    tick will be however many ticks fit until xmax.

    :data:`x_ticks_minor` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    x_grid = BooleanProperty(True)
    """Determines whether the x-axis has tick marks or a full grid.

    If :data:`x_ticks_major` is non-zero, then if x_grid is False tick marks
    will be displayed at every major tick. If x_grid is True, instead of ticks,
    a vertical line will be displayed at every major tick.

    :data:`x_grid` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    x_grid_label = BooleanProperty(True)
    """Whether labels should be displayed beneath each major tick. If true,
    each major tick will have a label containing the axis value.

    :data:`x_grid_label` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to False.
    """

    xlabel = StringProperty("")
    """The label for the x-axis. If not empty it is displayed in the center of
    the axis.

    :data:`xlabel` is a :class:`~kivy.properties.StringProperty`,
    defaults to ''.
    """

    ymin = NumericProperty(0.0)
    """The y-axis minimum value.

    :data:`ymin` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    ymax = NumericProperty(100.0)
    """The y-axis maximum value, larger than ymin.

    :data:`ymax` is a :class:`~kivy.properties.NumericProperty`, defaults to 0.
    """

    y_ticks_major = BoundedNumericProperty(0, min=0)
    """Distance between major tick marks. See :data:`x_ticks_major`.

    :data:`y_ticks_major` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    y_ticks_minor = BoundedNumericProperty(0, min=0)
    """The number of sub-intervals that divide ticks_major.
    See :data:`x_ticks_minor`.

    :data:`y_ticks_minor` is a
    :class:`~kivy.properties.BoundedNumericProperty`, defaults to 0.
    """

    y_grid = BooleanProperty(True)
    """Determines whether the y-axis has tick marks or a full grid. See
    :data:`x_grid`.

    :data:`y_grid` is a :class:`~kivy.properties.BooleanProperty`, defaults
    to False.
    """

    y_grid_label = BooleanProperty(True)
    """Whether labels should be displayed beneath each major tick. If true,
    each major tick will have a label containing the axis value.

    :data:`y_grid_label` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to False.
    """

    ylabel = StringProperty("")
    """The label for the y-axis. If not empty it is displayed in the center of
    the axis.

    :data:`ylabel` is a :class:`~kivy.properties.StringProperty`,
    defaults to ''.
    """

    padding = NumericProperty("5dp")
    """Padding distances between the labels, axes titles and graph, as
    well between the widget and the objects near the boundaries.

    :data:`padding` is a :class:`~kivy.properties.NumericProperty`, defaults
    to 5dp.
    """

    font_size = NumericProperty("15sp")
    """Font size of the labels.

    :data:`font_size` is a :class:`~kivy.properties.NumericProperty`, defaults
    to 15sp.
    """

    x_ticks_angle = NumericProperty(0)
    """Rotate angle of the x-axis tick marks.

    :data:`x_ticks_angle` is a :class:`~kivy.properties.NumericProperty`,
    defaults to 0.
    """

    precision = StringProperty("%g")
    """Determines the numerical precision of the tick mark labels. This value
    governs how the numbers are converted into string representation. Accepted
    values are those listed in Python's manual in the
    "String Formatting Operations" section.

    :data:`precision` is a :class:`~kivy.properties.StringProperty`, defaults
    to '%g'.
    """

    draw_border = BooleanProperty(True)
    """Whether a border is drawn around the canvas of the graph where the
    plots are displayed.

    :data:`draw_border` is a :class:`~kivy.properties.BooleanProperty`,
    defaults to True.
    """

    plots = ListProperty([])
    """Holds a list of all the plots in the graph. To add and remove plots
    from the graph use :data:`add_plot` and :data:`add_plot`. Do not add
    directly edit this list.

    :data:`plots` is a :class:`~kivy.properties.ListProperty`,
    defaults to [].
    """

    view_size = ObjectProperty((0, 0))
    """The size of the graph viewing area - the area where the plots are
    displayed, excluding labels etc.
    """

    view_pos = ObjectProperty((0, 0))
    """The pos of the graph viewing area - the area where the plots are
    displayed, excluding labels etc. It is relative to the graph's pos.
    """


class KivyWaveformCodeGraph(MultiGraph):
    def __init__(self):
        super().__init__()

    def do_plot(self):
        while True:
            try:
                dat = pickle.load(sys.stdin.buffer)
                q.put(dat)
            except EOFError:
                break

    def read_queue(self, dt):
        if q.empty():
            # print("empty")
            pass

        else:
            while not q.empty():
                item = q.get()
            # print(item)

            wave = item["wave"]
            yrange = item["yrange"]
            channels = item["channels"]
            trange = item["trange"]

            for ch in range(channels):
                while len(self.plots) <= ch:
                    plot = LinePlot()
                    self.add_plot(plot, channel=ch)

                plot = self.plots[ch]
                plot.points = enumerate((wave[ch, :] / yrange[1]).tolist(), trange[0])

            self.xmin, self.xmax = (trange[0], trange[1])
            self.ymin, self.ymax = (-1.0, 1.0)


class KivyWaveformCodeApp(App):
    def on_start(self):
        threading.Thread(target=self.root.do_plot).start()
        Clock.schedule_interval(self.root.read_queue, 0.001)

    def build(self):
        self.title = "HARK Python Waveform"
        g = KivyWaveformCodeGraph()
        g.xlabel = "Frames[sample]"
        g.ylabel = "Amp./Channels"
        g.x_ticks_major = 5000
        g.x_ticks_minor = 10
        g.x_grid = True
        g.y_grid = True
        g.x_grid_label = True
        g.y_grid_label = True
        g.ymin = -1.0
        g.ymax = 1.0
        return g


if __name__ == "__main__":
    KivyWaveformCodeApp().run()
