#! /usr/bin/env python
# -*- coding: utf-8 -*-

import os
import pickle
import sys
from subprocess import PIPE, Popen

import numpy

from hark import node


def cartesian2polar(xyz, mode="AZIMUTH"):
    """
    cartesian2polar(x, y, z)
    converts coordinate from (x, y, z) to (r, theta, phi)
    """
    x, y, z = xyz
    r = numpy.sqrt(numpy.sum(numpy.square(numpy.array(xyz))))
    theta = numpy.arctan2(z, numpy.sqrt(numpy.sum(numpy.square(numpy.array([x, y])))))
    phi = numpy.arctan2(y, x)

    return (r, theta, phi)


def r2d(r):
    return 180.0 / numpy.pi * r


class plotQuickSourceKivy(node.PythonNode):
    GUI_SCRIPT_NAME = "plotQuickSourceKivyGui.py"

    def __init__(self):
        super().__init__(["SOURCES", ("WINDOW_LENGTH", 300), ("UPDATE_INTERVAL", 50)], ["OUTPUT"])

        script_path = os.path.join(
            os.path.dirname(os.path.abspath(__file__)), self.GUI_SCRIPT_NAME
        )
        python_exec = sys.executable
        self.proc = Popen([python_exec, script_path], stdin=PIPE)

        self.points = {}
        self.windowlen = 300

        self.interval = 50
        self.count = 0

    def __del__(self):
        self.proc.kill()

    def update(self, data):
        pickle.dump(data, self.proc.stdin)

    def gendata(self, sources):
        if isinstance(sources, list):
            for src in sources:
                self.points.setdefault(src.id, [])
                self.points[src.id].append((self.count, r2d(cartesian2polar(src.x)[2])))

        if self.count % self.interval == 0:
            for id in list(self.points.keys()):
                pts = self.points[id]
                if pts[0][0] < self.count - self.windowlen:
                    cdel = self.count - self.windowlen - pts[0][0]
                    del self.points[id][:cdel]

                    if len(self.points[id]) == 0:
                        del self.points[id]

            self.update(
                {
                    "points": self.points,
                    "trange": (self.count - self.windowlen, self.count),
                }
            )

    def forward(self, **kwargs):
        if self.count == 0:
            # set Parameters
            if "WINDOW_LENGTH" in kwargs:
                if isinstance(kwargs["WINDOW_LENGTH"], int):
                    if kwargs["WINDOW_LENGTH"] > 0:
                        self.windowlen = kwargs["WINDOW_LENGTH"]
                    else:
                        print("[Error] The parameter WINDOW_LENGTH of the plotQuickSourceKivy node must be a positive value.")
                        raise ValueError
            if "UPDATE_INTERVAL" in kwargs:
                if isinstance(kwargs["UPDATE_INTERVAL"], int):
                    if kwargs["UPDATE_INTERVAL"] > 0:
                        self.interval = kwargs["UPDATE_INTERVAL"]
                    else:
                        print("[Error] The parameter UPDATE_INTERVAL of the plotQuickSourceKivy node must be a positive value.")
                        raise ValueError
            # check input terminal
            # if not isinstance(kwargs["SOURCES"], list):
            # if len(kwargs["SOURCES"]) > 0:
            #        try:
            #            raise AttributeError("There is an error in the network file. The type of input data to the plotQuickSourceKivy node must be Vector<Source>.")
            #        except AttributeError as e:
            #            print(e)
        self.gendata(kwargs["SOURCES"])

        self.count += 1
        return {"OUTPUT": kwargs["SOURCES"]}
