#! /usr/bin/env python
# -*- coding: utf-8 -*-

import os
import pickle
import sys
from subprocess import PIPE, Popen

import numpy

from hark import node


class plotQuickMusicSpecKivy(node.PythonNode):
    GUI_SCRIPT_NAME = "plotQuickMusicSpecKivyGui.py"

    def __init__(self):
        super().__init__(["SPECTRUM", ("WINDOW_LENGTH", 100), ("UPDATE_INTERVAL", 1), ("MAX_LIMIT", -1), ("MIN_LIMIT", -1)], ["OUTPUT"])

        script_path = os.path.join(
            os.path.dirname(os.path.abspath(__file__)), self.GUI_SCRIPT_NAME
        )
        python_exec = sys.executable
        self.proc = Popen([python_exec, script_path], stdin=PIPE)

        self.spectrum = numpy.array([])
        self.windowlen = 100
        self.max_limit = -1
        self.min_limit = -1

        self.interval = 1
        self.count = 0

    def __del__(self):
        self.proc.kill()

    def update(self, data):
        pickle.dump(data, self.proc.stdin)

    def gendata(self, data):
        tmp = data[numpy.newaxis, :]
        if tmp.sum() == 0:
            return

        if len(self.spectrum) == 0:
            self.spectrum = numpy.zeros(((self.windowlen - 1), len(data)))
        self.spectrum = numpy.r_[self.spectrum, tmp]
        if self.max_limit >= 0:
            self.spectrum[self.spectrum > self.max_limit] = self.max_limit
        if self.min_limit >= 0:
            self.spectrum[self.spectrum < self.min_limit] = self.min_limit

        if len(self.spectrum) > self.windowlen:
            self.spectrum = self.spectrum[-self.windowlen :, :]

        if self.count % self.interval == 0:
            self.update({"spectrum": self.spectrum, "offset": self.count})

    def forward(self, **kwargs):
        if self.count == 0:
            # set Parameters
            if "WINDOW_LENGTH" in kwargs:
                if isinstance(kwargs["WINDOW_LENGTH"], int):
                    if kwargs["WINDOW_LENGTH"] > 0:
                        self.windowlen = kwargs["WINDOW_LENGTH"]
                    else:
                        print("[Error] The parameter WINDOW_LENGTH of the plotQuickMusicSpecKivy node must be a positive value.")
                        raise ValueError
            if "UPDATE_INTERVAL" in kwargs:
                if isinstance(kwargs["UPDATE_INTERVAL"], int):
                    if kwargs["UPDATE_INTERVAL"] > 0:
                        self.interval = kwargs["UPDATE_INTERVAL"]
                    else:
                        print("[Error] The parameter UPDATE_INTERVAL of the plotQuickMusicSpecKivy node must be a positive value.")
                        raise ValueError
            if "MIN_LIMIT" in kwargs:
                if isinstance(kwargs["MIN_LIMIT"], int):
                    # In internal processing, all negative values ​​mean -1 (disabled).
                    if kwargs["MIN_LIMIT"] < 0:
                        self.min_limit = -1
                    else:
                        self.min_limit = kwargs["MIN_LIMIT"]
            if "MAX_LIMIT" in kwargs:
                if isinstance(kwargs["MAX_LIMIT"], int):
                    # In internal processing, all negative values ​​mean -1 (disabled).
                    if kwargs["MAX_LIMIT"] < 0:
                        self.max_limit = -1
                    elif kwargs["MAX_LIMIT"] > self.min_limit:
                        self.max_limit = kwargs["MAX_LIMIT"]
                    else:
                        print("[Error] The plotQuickMusicSpecKivy node parameter MAX_LIMIT must be greater than or equal to the parameter MIN_LIMIT.")
                        raise ValueError
            # check input terminal
            #if not isinstance(kwargs["SPECTRUM"], numpy.ndarray):
            #    try:
            #        raise AttributeError("There is an error in the network file. The type of input data to the plotQuickMUSICSpecKivy node must be Vector<float>.")
            #    except AttributeError as e:
            #        print(e)
        self.gendata(kwargs["SPECTRUM"])

        self.count += 1
        return {"OUTPUT": kwargs["SPECTRUM"]}
